package com.nimbusds.common.servlet;


import java.io.IOException;
import java.util.Properties;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.codahale.metrics.servlets.AdminServlet;
import com.nimbusds.common.config.ConfigurationException;
import com.nimbusds.common.oauth2.MasterAccessTokenValidator;
import com.nimbusds.common.oauth2.SHA256BasedAccessTokenValidator;
import com.thetransactioncompany.util.PropertyParseException;
import com.thetransactioncompany.util.PropertyRetriever;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Monitor servlet for exposing Dropwizard metrics and health checks, requires
 * an OAuth 2.0 bearer token for access.
 *
 * <p>The access token is defined by a
 * {@link Configuration#API_TOKEN_SHA256_PROPERTY_NAME Java property} obtained
 * from 1) system properties or from 2) a properties file specified by
 * {@link com.nimbusds.common.servlet.MonitorLauncher#CONFIG_CTX_PARAMETER_NAME
 * servlet context parameter}.
 */
public class MonitorServlet extends AdminServlet {
	
	
	/**
	 * The monitor servlet access token configuration.
	 */
	public static class Configuration {
		
		
		/**
		 * The property name for the SHA-256 hash of the API access
		 * token.
		 */
		public static final String API_TOKEN_SHA256_PROPERTY_NAME = "monitor.apiAccessTokenSHA256";
		
		
		/**
		 * The property name for the SHA-256 hash of the secondary API
		 * access token.
		 */
		public static final String SECONDARY_API_TOKEN_SHA256_PROPERTY_NAME = "monitor.secondaryAPIAccessTokenSHA256";
		
		
		/**
		 * The SHA-256 hash (in hexadecimal format) of the monitor API
		 * access token, if {@code null} the API is disabled.
		 */
		public final String apiAccessTokenSHA256;
		
		
		/**
		 * The SHA-256 hash (in hexadecimal format) of the optional
		 * secondary monitor API access token, {@code null} if none.
		 */
		public final String secondaryAPIAccessTokenSHA256;
		
		
		/**
		 * Creates a new monitor servlet access token configuration.
		 *
		 * @param properties The configuration properties, can be
		 *                   overridden with system properties.
		 *
		 * @throws ConfigurationException On a invalid configuration
		 *                                property.
		 */
		public Configuration(final Properties properties)
			throws ConfigurationException {
			
			if (System.getProperties().containsKey(API_TOKEN_SHA256_PROPERTY_NAME)) {
				// Override with system property
				properties.setProperty(API_TOKEN_SHA256_PROPERTY_NAME, System.getProperty(API_TOKEN_SHA256_PROPERTY_NAME));
			}
			
			if (System.getProperties().containsKey(SECONDARY_API_TOKEN_SHA256_PROPERTY_NAME)) {
				// Override
				properties.setProperty(SECONDARY_API_TOKEN_SHA256_PROPERTY_NAME, System.getProperty(SECONDARY_API_TOKEN_SHA256_PROPERTY_NAME));
			}
			
			
			PropertyRetriever pr = new PropertyRetriever(properties);
			
			try {
				apiAccessTokenSHA256 = pr.getOptString(API_TOKEN_SHA256_PROPERTY_NAME, null);
				secondaryAPIAccessTokenSHA256 = pr.getOptString(SECONDARY_API_TOKEN_SHA256_PROPERTY_NAME, null);
				
			} catch (PropertyParseException e) {
				throw new ConfigurationException(e.getMessage() + ": Property: " + e.getPropertyKey(), e);
			}
		}
	}


	/**
	 * The access token validator.
	 */
	protected MasterAccessTokenValidator tokenValidator;
	
	
	/**
	 * Creates a new access token validator.
	 *
	 * @param config The servlet configuration, used to retrieve the
	 *               monitor configuration properties.
	 *
	 * @return The basic access token validator.
	 *
	 * @throws ServletException If a configuration property is missing or
	 *                          invalid.
	 */
	static MasterAccessTokenValidator createAccessTokenValidator(final ServletConfig config)
		throws ServletException {
		
		Logger log = LogManager.getLogger("MAIN");
		
		Configuration monitorConfig;
		
		try {
			Properties props = ResourceRetriever.getProperties(
				config.getServletContext(),
				MonitorLauncher.CONFIG_CTX_PARAMETER_NAME,
				log);
			
			monitorConfig = new Configuration(props);
			
		} catch (Exception e) {
			log.error(e.getMessage(), e);
			throw new ServletException(e.getMessage(), e);
		}
		
		if (monitorConfig.secondaryAPIAccessTokenSHA256 != null) {
			return new SHA256BasedAccessTokenValidator(monitorConfig.apiAccessTokenSHA256, monitorConfig.secondaryAPIAccessTokenSHA256);
		} else {
			return new SHA256BasedAccessTokenValidator(monitorConfig.apiAccessTokenSHA256);
		}
	}


	@Override
	public void init(final ServletConfig config)
		throws ServletException {

		super.init(config);
		
		tokenValidator = createAccessTokenValidator(config);
		
		LogManager.getLogger("MAIN").info("[CM7110] Loaded monitor API servlet");
	}


	@Override
	protected void doGet(final HttpServletRequest req,
			     final HttpServletResponse resp)
		throws ServletException, IOException {

		if (! tokenValidator.validateBearerAccessToken(req, resp)) {
			return; // invalid or missing token, or web API disabled
		}

		super.doGet(req, resp);
	}


	@Override
	protected void service(final HttpServletRequest req,
			       final HttpServletResponse resp)
		throws ServletException, IOException {

		if (! tokenValidator.validateBearerAccessToken(req, resp)) {
			return; // invalid or missing token, or web API disabled
		}

		super.service(req, resp);
	}
}
