package com.nimbusds.common.spi;


import java.util.HashSet;
import java.util.Iterator;
import java.util.ServiceLoader;
import java.util.Set;


/**
 * Service loader utility.
 */
public class ServiceLoaderUtility {
	
	
	/**
	 * Loads a single Service Provider Interface (SPI) implementation.
	 *
	 * @param tClass      The SPI. Not {@code null}.
	 * @param defaultImpl The default implementation, {@code null} if not
	 *                    specified.
	 *
	 * @return The loaded SPI implementation or the default one.
	 *
	 * @throws RuntimeException If more than one SPI implementation was
	 *                          found.
	 */
	public static <T> T loadSingle(final Class<T> tClass, final T defaultImpl) {
		
		T impl = null;
		
		Iterator<T> it = ServiceLoader.load(tClass).iterator();
		
		if (it.hasNext()) {
			
			impl = it.next();
			
			if (it.hasNext())
				throw new RuntimeException("More than one " + tClass.getName() + " SPI implementation found");
		}
		
		if (impl == null) impl = defaultImpl;
		
		return impl;
	}
	
	
	/**
	 * Loads multiple Service Provider Interface (SPI) implementations.
	 *
	 * @param tClass      The SPI. Not {@code null}.
	 * @param defaultImpl The default implementation, {@code null} if not
	 *                    specified.
	 *
	 * @return The loaded SPI implementations, if none were loaded and a
	 *         default was specified, a singleton set with it.
	 */
	public static <T> Set<T> loadMultiple(final Class<T> tClass, final T defaultImpl) {
		
		Set<T> impls = new HashSet<>();
		
		for (T t : ServiceLoader.load(tClass)) {
			impls.add(t);
		}
		
		if (impls.isEmpty() && defaultImpl != null) {
			impls.add(defaultImpl);
		}
		
		return impls;
	}
}
