package com.nimbusds.common.oauth2;


import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.WebApplicationException;

import com.thetransactioncompany.util.PropertyParseException;
import com.thetransactioncompany.util.PropertyRetriever;
import net.jcip.annotations.ThreadSafe;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;


/**
 * SHA-256 based access token validator. The expected access tokens are
 * configured as their SHA-256 hashes, to prevent accidental leaks into logs,
 * etc. Supports servlet-based and JAX-RS based web applications.
 */
@ThreadSafe
public class SHA256BasedAccessTokenValidator extends AbstractAccessTokenValidator {
	
	
	/**
	 * The minimum acceptable access token length.
	 */
	public static final int MIN_TOKEN_LENGTH = 32;
	
	
	/**
	 * Creates a new access token validator.
	 *
	 * @param tokenHash The Bearer access token SHA-256 hash (in hex). If
	 *                  {@code null} access to the web API will be
	 *                  disabled.
	 */
	public SHA256BasedAccessTokenValidator(final String tokenHash) {
		
		this(new String[]{tokenHash});
	}
	
	/**
	 * Creates a new access token validator.
	 *
	 * @param tokenHashes The Bearer access token SHA-256 hashes (in hex).
	 *                    If {@code null} access to the web API will be
	 *                    disabled.
	 */
	public SHA256BasedAccessTokenValidator(final String ... tokenHashes) {
		
		for (String hash: tokenHashes) {
			if (hash == null) continue;
			try {
				expectedTokenHashes.add(Hex.decodeHex(hash.toCharArray()));
			} catch (DecoderException e) {
				throw new IllegalArgumentException("Invalid hex: " + hash);
			}
		}
		
		hashSalt = null;
	}
	
	
	/**
	 * Creates a new access token validator.
	 *
	 * @param tokenHash             The main Bearer access token SHA-256
	 *                              hash (in hex). If {@code null} access
	 *                              to the web API will be disabled.
	 * @param additionalTokenHashes Additional Bearer access token SHA-256
	 *                              hashes (in hex), empty or {@code null}
	 *                              if none.
	 */
	public SHA256BasedAccessTokenValidator(final String tokenHash, final List<String> additionalTokenHashes) {
	
		if (tokenHash == null) {
			return;
		}
		
		try {
			expectedTokenHashes.add(Hex.decodeHex(tokenHash.toCharArray()));
			
			if (additionalTokenHashes != null) {
				for (String hash: additionalTokenHashes) {
					if (hash != null) {
						expectedTokenHashes.add(Hex.decodeHex(hash.toCharArray()));
					}
				}
			}
			
		} catch (DecoderException e) {
			throw new IllegalArgumentException("Invalid hex");
		}
	}
	
	
	/**
	 * Creates a new access token validator from the specified properties.
	 *
	 * @param props                        The properties. Must not be
	 *                                     {@code null}.
	 * @param propertyName                 The property name for the main
	 *                                     Bearer access token SHA-256 hash
	 *                                     (in hex). If {@code null} access
	 * 	                               to the web API will be disabled.
	 * 	                               Must not be {@code null}.
	 * @param additionalPropertyNamePrefix The property name prefix for the
	 *                                     additional Bearer access token
	 *                                     SHA-256 hashes (in hex),
	 *                                     {@code null} if not used.
	 *
	 * @return The access token validator.
	 *
	 * @throws PropertyParseException If parsing failed.
	 */
	public static SHA256BasedAccessTokenValidator from(final Properties props,
							   final String propertyName,
							   final String additionalPropertyNamePrefix)
		throws PropertyParseException {
		
		PropertyRetriever pr = new PropertyRetriever(props);
		
		String tokenHash = pr.getOptString(propertyName, null);
		
		if (additionalPropertyNamePrefix == null) {
			return new SHA256BasedAccessTokenValidator(tokenHash);
		}
		
		List<String> additionalTokenHashes = pr.getOptStringListMulti(additionalPropertyNamePrefix, Collections.emptyList());
		
		return new SHA256BasedAccessTokenValidator(tokenHash, additionalTokenHashes);
	}
	
	
	@Override
	public void validateBearerAccessToken(final String authzHeader)
		throws WebApplicationException {
		
		// Web API disabled?
		if (accessIsDisabled()) {
			throw WEB_API_DISABLED.toWebAppException();
		}
		
		if (StringUtils.isBlank(authzHeader)) {
			throw MISSING_BEARER_TOKEN.toWebAppException();
		}
		
		BearerAccessToken receivedToken;
		
		try {
			receivedToken = BearerAccessToken.parse(authzHeader);
			
		} catch (ParseException e) {
			throw MISSING_BEARER_TOKEN.toWebAppException();
		}
		
		if (null != log) {
			log.trace("[CM3000] Validating bearer access token: {}", TokenAbbreviator.abbreviate(receivedToken));
		}
		
		// Check min length
		if (receivedToken.getValue().length() < MIN_TOKEN_LENGTH) {
			throw INVALID_BEARER_TOKEN.toWebAppException();
		}
		
		if (isValid(receivedToken)) {
			return; // pass
		}
		
		throw INVALID_BEARER_TOKEN.toWebAppException();
	}
	
	
	@Override
	public boolean validateBearerAccessToken(final HttpServletRequest servletRequest,
						 final HttpServletResponse servletResponse)
		throws IOException {
		
		// Web API disabled?
		if (accessIsDisabled()) {
			WEB_API_DISABLED.apply(servletResponse);
			return false;
		}
		
		BearerAccessToken receivedToken;
		
		if (servletRequest.getHeader("Authorization") != null) {
			
			String authzHeaderValue = servletRequest.getHeader("Authorization");
			
			if (StringUtils.isBlank(authzHeaderValue)) {
				MISSING_BEARER_TOKEN.apply(servletResponse);
				return false;
			}
			
			try {
				receivedToken = BearerAccessToken.parse(authzHeaderValue);
				
			} catch (ParseException e) {
				MISSING_BEARER_TOKEN.apply(servletResponse);
				return false;
			}
			
		} else if (servletRequest.getParameter("access_token") != null) {
			
			String accessTokenValue = servletRequest.getParameter("access_token");
			
			if (StringUtils.isBlank(accessTokenValue)) {
				MISSING_BEARER_TOKEN.apply(servletResponse);
				return false;
			}
			
			receivedToken = new BearerAccessToken(accessTokenValue);
		} else {
			MISSING_BEARER_TOKEN.apply(servletResponse);
			return false;
		}
		
		if (null != log) {
			log.trace("[CM3000] Validating bearer access token: {}", TokenAbbreviator.abbreviate(receivedToken));
		}
		
		// Check min length
		if (receivedToken.getValue().length() < MIN_TOKEN_LENGTH) {
			INVALID_BEARER_TOKEN.apply(servletResponse);
			return false;
		}
		
		// Compare hashes
		if (isValid(receivedToken)) {
			return true; // pass
		}
		
		INVALID_BEARER_TOKEN.apply(servletResponse);
		return false;
	}
}
