/*
 * Decompiled with CFR 0.152.
 */
package com.norconex.commons.lang.file;

import com.norconex.commons.lang.Sleeper;
import com.norconex.commons.lang.StringUtil;
import com.norconex.commons.lang.file.IFileVisitor;
import com.norconex.commons.lang.io.IInputStreamFilter;
import com.norconex.commons.lang.io.ReverseFileInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.LinkedList;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.CharUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public final class FileUtil {
    private static final Logger LOG = LogManager.getLogger(FileUtil.class);
    private static final int MAX_FILE_OPERATION_ATTEMPTS = 10;

    private FileUtil() {
    }

    public static String toSafeFileName(String unsafeFileName) {
        if (unsafeFileName == null) {
            return null;
        }
        StringBuilder b = new StringBuilder();
        for (int i = 0; i < unsafeFileName.length(); ++i) {
            char ch = unsafeFileName.charAt(i);
            if (CharUtils.isAsciiAlphanumeric((char)ch) || ch == '-' || ch == '.') {
                b.append(ch);
                continue;
            }
            b.append('_');
            b.append((int)ch);
            b.append('_');
        }
        return b.toString();
    }

    public static String fromSafeFileName(String safeFileName) {
        if (safeFileName == null) {
            return null;
        }
        StringBuilder b = new StringBuilder();
        for (int i = 0; i < safeFileName.length(); ++i) {
            char ch = safeFileName.charAt(i);
            if (ch == '_') {
                String intVal = StringUtils.substring((String)safeFileName, (int)(i + 1), (int)StringUtils.indexOf((CharSequence)safeFileName, (int)95, (int)(i + 1)));
                b.append((char)NumberUtils.toInt((String)intVal));
                i += intVal.length() + 1;
                continue;
            }
            b.append(ch);
        }
        return b.toString();
    }

    public static void moveFileToDir(File sourceFile, File targetDir) throws IOException {
        if (sourceFile == null || !sourceFile.isFile()) {
            throw new IOException("Source file is not valid: " + sourceFile);
        }
        if (targetDir == null || targetDir.exists() && !targetDir.isDirectory()) {
            throw new IOException("Target directory is not valid:" + targetDir);
        }
        if (!targetDir.exists()) {
            FileUtils.forceMkdir((File)targetDir);
        }
        String fileName = sourceFile.getName();
        File targetFile = new File(targetDir, fileName);
        FileUtil.moveFile(sourceFile, targetFile);
    }

    public static void moveFile(File sourceFile, File targetFile) throws IOException {
        int failure;
        if (!FileUtil.isFile(sourceFile)) {
            throw new IOException("Source file is not a file or is not valid: " + sourceFile);
        }
        if (targetFile == null || targetFile.exists() && !targetFile.isFile()) {
            throw new IOException("Target file is not a file or is not valid: " + targetFile);
        }
        Throwable ex = null;
        for (failure = 0; failure < 10 && (targetFile.exists() && !targetFile.delete() || !sourceFile.renameTo(targetFile)); ++failure) {
            Sleeper.sleepSeconds(1);
        }
        if (failure >= 10) {
            throw new IOException("Could not move \"" + sourceFile + "\" to \"" + targetFile + "\".", ex);
        }
    }

    public static void delete(File file) throws IOException {
        if (file == null || !file.exists()) {
            return;
        }
        boolean success = false;
        int failure = 0;
        while (!success && failure < 10) {
            if (file.exists() && !FileUtils.deleteQuietly((File)file)) {
                ++failure;
                System.gc();
                Sleeper.sleepSeconds(1);
                continue;
            }
            success = true;
        }
        if (!success) {
            throw new IOException("Could not delete \"" + file + "\".");
        }
    }

    public static int deleteEmptyDirs(File parentDir) {
        return FileUtil.deleteEmptyDirs(parentDir, null);
    }

    public static int deleteEmptyDirs(File parentDir, final Date date) {
        final MutableInt dirCount = new MutableInt(0);
        FileUtil.visitEmptyDirs(parentDir, new IFileVisitor(){

            @Override
            public void visit(File file) {
                if (date == null || FileUtils.isFileOlder((File)file, (Date)date)) {
                    String[] children = file.list();
                    if (file.isDirectory() && (children == null || children.length == 0)) {
                        try {
                            FileUtil.delete(file);
                            dirCount.increment();
                        }
                        catch (IOException e) {
                            LOG.error((Object)("Could not be delete directory: " + file), (Throwable)e);
                        }
                    }
                }
            }
        });
        return dirCount.intValue();
    }

    public static File createDirsForFile(File file) throws IOException {
        File parent = file.getParentFile();
        if (parent != null) {
            FileUtils.forceMkdir((File)parent);
            return parent;
        }
        return new File("/");
    }

    public static void visitAllDirsAndFiles(File dir, IFileVisitor visitor) {
        FileUtil.visitAllDirsAndFiles(dir, visitor, null);
    }

    public static void visitAllDirsAndFiles(File dir, IFileVisitor visitor, FileFilter filter) {
        File[] children;
        visitor.visit(dir);
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory() && (children = dir.listFiles(filter)) != null) {
            for (int i = 0; i < children.length; ++i) {
                FileUtil.visitAllDirsAndFiles(children[i], visitor, filter);
            }
        }
    }

    public static void visitEmptyDirs(File dir, IFileVisitor visitor) {
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory()) {
            String[] children = dir.list();
            if (children == null || children.length == 0) {
                visitor.visit(dir);
            } else {
                for (int i = 0; i < children.length; ++i) {
                    FileUtil.visitAllDirs(new File(dir, children[i]), visitor);
                }
            }
        }
    }

    public static void visitEmptyDirs(File dir, IFileVisitor visitor, FileFilter filter) {
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory()) {
            File[] children = dir.listFiles(filter);
            if (children == null || children.length == 0) {
                visitor.visit(dir);
            } else {
                for (int i = 0; i < children.length; ++i) {
                    FileUtil.visitAllDirs(children[i], visitor, filter);
                }
            }
        }
    }

    public static void visitAllDirs(File dir, IFileVisitor visitor) {
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory()) {
            visitor.visit(dir);
            String[] children = dir.list();
            if (children != null) {
                for (int i = 0; i < children.length; ++i) {
                    FileUtil.visitAllDirs(new File(dir, children[i]), visitor);
                }
            }
        }
    }

    public static void visitAllDirs(File dir, IFileVisitor visitor, FileFilter filter) {
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory()) {
            visitor.visit(dir);
            File[] children = dir.listFiles(filter);
            if (children != null) {
                for (int i = 0; i < children.length; ++i) {
                    FileUtil.visitAllDirs(children[i], visitor, filter);
                }
            }
        }
    }

    public static void visitAllFiles(File dir, IFileVisitor visitor) {
        FileUtil.visitAllFiles(dir, visitor, null);
    }

    public static void visitAllFiles(File dir, IFileVisitor visitor, FileFilter filter) {
        if (!dir.exists()) {
            return;
        }
        if (dir.isDirectory()) {
            String[] children = dir.list();
            if (children != null) {
                for (int i = 0; i < children.length; ++i) {
                    FileUtil.visitAllFiles(new File(dir, children[i]), visitor, filter);
                }
            }
        } else if (filter == null || filter.accept(dir)) {
            visitor.visit(dir);
        }
    }

    public static String[] head(File file, int numberOfLinesToRead) throws IOException {
        return FileUtil.head(file, StandardCharsets.UTF_8.toString(), numberOfLinesToRead);
    }

    public static String[] head(File file, String encoding, int numberOfLinesToRead) throws IOException {
        return FileUtil.head(file, encoding, numberOfLinesToRead, true);
    }

    public static String[] head(File file, String encoding, int numberOfLinesToRead, boolean stripBlankLines) throws IOException {
        return FileUtil.head(file, encoding, numberOfLinesToRead, stripBlankLines, null);
    }

    public static String[] head(File file, String encoding, int numberOfLinesToRead, boolean stripBlankLines, IInputStreamFilter filter) throws IOException {
        FileUtil.assertFile(file);
        FileUtil.assertNumOfLinesToRead(numberOfLinesToRead);
        LinkedList<String> lines = new LinkedList<String>();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(file), encoding));){
            int remainingLinesToRead = numberOfLinesToRead;
            String line = "";
            while (line != null && remainingLinesToRead-- > 0) {
                line = reader.readLine();
                if (!stripBlankLines || StringUtils.isNotBlank((CharSequence)line)) {
                    if (filter != null && filter.accept(line)) {
                        lines.addFirst(line);
                        continue;
                    }
                    ++remainingLinesToRead;
                    continue;
                }
                ++remainingLinesToRead;
            }
        }
        return lines.toArray(ArrayUtils.EMPTY_STRING_ARRAY);
    }

    public static String[] tail(File file, int numberOfLinesToRead) throws IOException {
        return FileUtil.tail(file, StandardCharsets.UTF_8.toString(), numberOfLinesToRead);
    }

    public static String[] tail(File file, String encoding, int numberOfLinesToRead) throws IOException {
        return FileUtil.tail(file, encoding, numberOfLinesToRead, true);
    }

    public static String[] tail(File file, String encoding, int numberOfLinesToRead, boolean stripBlankLines) throws IOException {
        return FileUtil.tail(file, encoding, numberOfLinesToRead, stripBlankLines, null);
    }

    public static String[] tail(File file, String encoding, int numberOfLinesToRead, boolean stripBlankLines, IInputStreamFilter filter) throws IOException {
        FileUtil.assertFile(file);
        FileUtil.assertNumOfLinesToRead(numberOfLinesToRead);
        LinkedList<String> lines = new LinkedList<String>();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)new ReverseFileInputStream(file), encoding));){
            String line;
            int remainingLinesToRead = numberOfLinesToRead;
            while ((line = reader.readLine()) != null) {
                if (remainingLinesToRead-- <= 0) {
                    break;
                }
                String newLine = StringUtils.reverse((String)line);
                if (!stripBlankLines || StringUtils.isNotBlank((CharSequence)line)) {
                    if (filter != null && filter.accept(newLine)) {
                        lines.addFirst(newLine);
                        continue;
                    }
                    ++remainingLinesToRead;
                    continue;
                }
                ++remainingLinesToRead;
            }
        }
        return lines.toArray(ArrayUtils.EMPTY_STRING_ARRAY);
    }

    public static File createDateDirs(File parentDir) throws IOException {
        return FileUtil.createDateDirs(parentDir, new Date());
    }

    public static File createDateDirs(File parentDir, Date date) throws IOException {
        return FileUtil.createDateTimeDirs(parentDir, date, "yyyy/MM/dd");
    }

    public static File createDateTimeDirs(File parentDir) throws IOException {
        return FileUtil.createDateTimeDirs(parentDir, new Date());
    }

    public static File createDateTimeDirs(File parentDir, Date dateTime) throws IOException {
        return FileUtil.createDateTimeDirs(parentDir, dateTime, "yyyy/MM/dd/HH/mm/ss");
    }

    public static File createURLDirs(File parentDir, URL url) throws IOException {
        return FileUtil.createURLDirs(parentDir, url, false);
    }

    public static File createURLDirs(File parentDir, String url) throws IOException {
        return FileUtil.createURLDirs(parentDir, url, false);
    }

    public static File createURLDirs(File parentDir, URL url, boolean truncate) throws IOException {
        if (url == null) {
            throw new IOException("URL cannot be null.");
        }
        return FileUtil.createURLDirs(parentDir, url.toString(), truncate);
    }

    public static File createURLDirs(File parentDir, String url, boolean truncate) throws IOException {
        String[] segs;
        if (parentDir == null) {
            throw new IOException("Parent directory cannot be null.");
        }
        if (url == null) {
            throw new IOException("URL cannot be null.");
        }
        if (parentDir.exists() && !parentDir.isDirectory()) {
            throw new IOException("Parent directory \"" + parentDir + "\" already exists and is not a directory.");
        }
        if (truncate && parentDir.getAbsolutePath().length() > 200) {
            throw new IOException("Parent directory \"" + parentDir + "\" is too long (must be 200 characters or less).");
        }
        StringBuilder b = new StringBuilder(parentDir.getAbsolutePath());
        for (String seg : segs = url.replaceFirst("://", "/").split("/")) {
            b.append("/").append(FileUtil.toSafeFileName(seg));
        }
        String path = b.toString();
        if (truncate) {
            path = StringUtil.truncateWithHash(path, 255, "_");
        }
        File urlFile = new File(path);
        FileUtil.createDirsForFile(urlFile);
        return urlFile;
    }

    private static File createDateTimeDirs(File parentDir, Date dateTime, String format) throws IOException {
        if (parentDir == null) {
            throw new IOException("Parent directory cannot be null.");
        }
        if (dateTime == null) {
            throw new IOException("Date cannot be null.");
        }
        if (parentDir.exists() && !parentDir.isDirectory()) {
            throw new IOException("Parent directory \"" + parentDir + "\" already exists and is not a directory.");
        }
        File dateDir = new File(parentDir.getAbsolutePath(), DateFormatUtils.format((Date)dateTime, (String)format));
        FileUtils.forceMkdir((File)dateDir);
        return dateDir;
    }

    private static void assertNumOfLinesToRead(int num) {
        if (num <= 0) {
            throw new IllegalArgumentException("Not a valid number to read: " + num);
        }
    }

    private static void assertFile(File file) throws IOException {
        if (!(file != null && file.exists() && file.isFile() && file.canRead())) {
            throw new IOException("Not a valid file: " + file);
        }
    }

    private static boolean isFile(File file) {
        return file != null && file.isFile();
    }
}

