/*
 * Decompiled with CFR 0.152.
 */
package com.oceanbase.tools.datamocker.core.write.output;

import com.oceanbase.tools.datamocker.core.write.output.ConnectionWrapper;
import com.oceanbase.tools.datamocker.model.config.model.DataBaseConfig;
import com.oceanbase.tools.datamocker.model.exception.MockerError;
import com.oceanbase.tools.datamocker.model.exception.MockerException;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import javax.sql.DataSource;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MockerDataSource
implements DataSource {
    private static final Logger log = LoggerFactory.getLogger(MockerDataSource.class);
    private final DataBaseConfig config;
    private final String jdbcUrl;
    private final Map<String, String> connectParams;
    private static final String DRIVER_CLASS_NAME = "com.alipay.oceanbase.obproxy.mysql.jdbc.Driver";
    private int minPoolSize = 10;
    private int maxPoolSize = 20;
    private int increaseStep = 5;
    private final Stack<MockerConnection> connectionPool = new Stack();
    private final List<MockerConnection> connectionInUse = new LinkedList<MockerConnection>();
    private final ReentrantLock lock = new ReentrantLock();
    private final Condition noMoreConnection = this.lock.newCondition();
    private final AtomicInteger connectionCounter = new AtomicInteger(0);

    public MockerDataSource(DataBaseConfig config, Map<String, String> connectParam) throws SQLException {
        this.validateJdbcConfig(config);
        this.config = config;
        this.connectParams = connectParam;
        this.jdbcUrl = this.generateJdbcUrl();
        this.initPool();
    }

    public MockerDataSource(DataBaseConfig config, int minPoolSize, int maxPoolSize, int increaseStep, Map<String, String> connectParam) throws SQLException {
        this.validateJdbcConfig(config);
        this.config = config;
        if (minPoolSize <= 0 || maxPoolSize <= 0 || increaseStep <= 0) {
            throw new MockerException(MockerError.PARAMETER_ERROR, "Min pool size, max pool size or increase step can not be equal to or less than zero");
        }
        if (minPoolSize > maxPoolSize) {
            throw new MockerException(MockerError.PARAMETER_ERROR, "Min pool size can not be bigger than max pool size");
        }
        this.minPoolSize = minPoolSize;
        this.increaseStep = increaseStep;
        this.maxPoolSize = maxPoolSize;
        this.connectParams = connectParam;
        this.jdbcUrl = this.generateJdbcUrl();
        this.initPool();
    }

    private void initPool() throws SQLException {
        StringBuilder username = new StringBuilder(this.config.getUser());
        if (StringUtils.isNotBlank((String)this.config.getTenant())) {
            username.append("@").append(this.config.getTenant());
        }
        if (StringUtils.isNotBlank((String)this.config.getCluster())) {
            username.append("#").append(this.config.getCluster());
        }
        ((MockerConnection)this.getConnection(username.toString(), this.config.getPassword())).closeConnection();
        this.refresh();
    }

    private void validateJdbcConfig(DataBaseConfig config) {
        Validate.notNull((Object)config, (String)"DataBase config can not be null for MockerDataSource#validate");
        Validate.notEmpty((String)config.getHost(), (String)"Host can not be blank for MockerDataSource#validate");
        Validate.notEmpty((String)config.getUser(), (String)"User can not be blank for MockerDataSource#validate");
        Validate.notNull((Object)config.getPassword(), (String)"Password can not be null for MockerDataSource#validate");
        Validate.notEmpty((String)config.getDefaultSchame(), (String)"DefaultSchemaName can not be blank for MockerDataSource#validate");
        Validate.notNull((Object)config.getPort(), (String)"Port can not be blank for MockerDataSource#validate");
    }

    private String generateJdbcUrl() {
        StringBuilder buffer = new StringBuilder("jdbc:oceanbase://");
        buffer.append(this.config.getHost()).append(":").append(this.config.getPort()).append("/").append(this.config.getDefaultSchame());
        if (this.connectParams != null) {
            Set<Map.Entry<String, String>> entrySet = this.connectParams.entrySet();
            String paramStr = entrySet.stream().map(stringStringEntry -> (String)stringStringEntry.getKey() + "=" + (String)stringStringEntry.getValue()).collect(Collectors.joining("&"));
            buffer.append("?").append(paramStr);
        }
        return buffer.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refresh() {
        this.lock.lock();
        try {
            int liveConnectionCount = this.connectionPool.size();
            Iterator<MockerConnection> iter = this.connectionInUse.iterator();
            while (iter.hasNext()) {
                MockerConnection mockerConnection = iter.next();
                if (!mockerConnection.isClosed() && mockerConnection.isValid(this.getLoginTimeout())) {
                    ++liveConnectionCount;
                    continue;
                }
                log.debug("Remove dead connection from using pool, connectionId={}", (Object)mockerConnection.connectionId);
                iter.remove();
            }
            StringBuilder username = new StringBuilder(this.config.getUser());
            if (StringUtils.isNotBlank((String)this.config.getTenant())) {
                username.append("@").append(this.config.getTenant());
            }
            if (StringUtils.isNotBlank((String)this.config.getCluster())) {
                username.append("#").append(this.config.getCluster());
            }
            if (liveConnectionCount < this.minPoolSize) {
                int interval = this.minPoolSize - liveConnectionCount;
                log.info("The number of surviving connections is less than the minimum number of connections, start adding connections, liveConnectionsCount={}, minPoolSize={}, triggeredThreadName={}", new Object[]{liveConnectionCount, liveConnectionCount + interval, Thread.currentThread().getName()});
                for (int i = 0; i < interval; ++i) {
                    this.connectionPool.push((MockerConnection)this.getConnection(username.toString(), this.config.getPassword()));
                }
            } else if (liveConnectionCount > this.maxPoolSize) {
                int count = Math.min(liveConnectionCount - this.maxPoolSize, this.connectionPool.size());
                log.info("The number of surviving connections is greater than the maximum number of connections, start to reduce the number of connections, liveConnectionsCount={}, maxPoolSize={}, triggeredThreadName={}", new Object[]{liveConnectionCount, liveConnectionCount - count, Thread.currentThread().getName()});
                for (int i = 0; i < count; ++i) {
                    try {
                        this.connectionPool.pop().close();
                        continue;
                    }
                    catch (SQLException e) {
                        log.error("Fail to close the connection", (Throwable)e);
                    }
                }
            } else {
                int realStep = Math.min(this.maxPoolSize - liveConnectionCount, this.increaseStep);
                log.info("The number of surviving connections is between the maximum and minimum values, and the connections begin to expand, liveConnectionsCount={}, targetPoolSize={}, triggeredThreadName={}", new Object[]{liveConnectionCount, liveConnectionCount + realStep, Thread.currentThread().getName()});
                for (int i = 0; i < realStep; ++i) {
                    this.connectionPool.push((MockerConnection)this.getConnection(username.toString(), this.config.getPassword()));
                }
            }
            if (this.connectionPool.size() > 0) {
                this.noMoreConnection.signalAll();
            }
        }
        catch (SQLException e) {
            log.error("Fail to refresh connection pool", (Throwable)e);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Connection getConnection() throws SQLException {
        this.lock.lock();
        try {
            if (this.connectionPool.size() == 0) {
                log.warn("There are no more connections available in the connection pool, start refreshing, triggeredThreadName={}", (Object)Thread.currentThread().getName());
                this.refresh();
                if (this.connectionPool.size() == 0) {
                    log.warn("No connection is released from the connection pool, the thread starts to wait, threadName={}", (Object)Thread.currentThread().getName());
                    this.noMoreConnection.await(3L, TimeUnit.SECONDS);
                    Connection connection = this.getConnection();
                    return connection;
                }
            }
            MockerConnection connection = this.connectionPool.pop();
            log.debug("Pop connection from pool, connectionId={}", (Object)connection.connectionId);
            if (!connection.isValid(this.getLoginTimeout()) || connection.isClosed()) {
                Connection connection2 = this.getConnection();
                return connection2;
            }
            this.connectionInUse.add(connection);
            log.debug("The thread gets the database connection from the connection pool successfully, connectionId={}, threadName={}, currentPoolSize={}, connectionInUseSize={}", new Object[]{connection.connectionId, Thread.currentThread().getName(), this.connectionPool.size(), this.connectionInUse.size()});
            MockerConnection mockerConnection = connection;
            return mockerConnection;
        }
        catch (InterruptedException e) {
            log.error("Thread waiting for connection pool read lock failed, threadName={}", (Object)Thread.currentThread().getName(), (Object)e);
            throw new SQLException(e.getMessage());
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean clear() {
        if (this.lock.tryLock()) {
            try {
                int count = this.connectionPool.size();
                log.info("The database connection pool will be closed, the free database connection needs to be closed, currentPoolSize={}", (Object)count);
                int clearSize = 0;
                for (int i = 0; i < count; ++i) {
                    try {
                        this.connectionPool.pop().closeConnection();
                        ++clearSize;
                        continue;
                    }
                    catch (SQLException e) {
                        log.error("Fail to close the connection", (Throwable)e);
                    }
                }
                log.info("Free database connection closed completely, clearConnectionCount={}, failedConnectionCount={}", (Object)clearSize, (Object)(count - clearSize));
                clearSize = 0;
                int closeSize = 0;
                int inUseCount = this.connectionInUse.size();
                log.info("The database connection in use will be closed, connectionInUseCount={}", (Object)inUseCount);
                for (int i = 0; i < inUseCount; ++i) {
                    try {
                        if (!this.connectionInUse.get(i).isClosed()) {
                            this.connectionInUse.get(i).closeConnection();
                            ++clearSize;
                            continue;
                        }
                        ++closeSize;
                        continue;
                    }
                    catch (SQLException e) {
                        log.error("Fail to close the connection in use", (Throwable)e);
                    }
                }
                log.info("The database connection pool is closed successfully, clearConnectionSize={}, closedByThreadConnectionSize={}, failedConnectionCount={}", new Object[]{clearSize, closeSize, inUseCount - clearSize - closeSize});
            }
            finally {
                this.lock.unlock();
            }
            return true;
        }
        return false;
    }

    @Override
    public Connection getConnection(String username, String password) throws SQLException {
        int id = this.connectionCounter.incrementAndGet();
        log.debug("Create connection, connectionId={} ", (Object)id);
        return new MockerConnection(this, DriverManager.getConnection(this.jdbcUrl, username, password), id, true);
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
        DriverManager.setLoginTimeout(seconds);
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return DriverManager.getLoginTimeout();
    }

    @Override
    public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException("not support for MockerDataSourceManager#getParentLogger method");
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("not support for MockerDataSourceManager#unwrap method");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("not support for MockerDataSourceManager#isWrapperFor method");
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        throw new SQLFeatureNotSupportedException("not support for MockerDataSourceManager#getLogWriter method");
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
        throw new SQLFeatureNotSupportedException("not support for MockerDataSourceManager#setLogWriter method");
    }

    static {
        try {
            DriverManager.setLoginTimeout(15);
            Class.forName(DRIVER_CLASS_NAME);
        }
        catch (ClassNotFoundException e) {
            log.error("Data source initialization failed", (Throwable)e);
        }
    }

    private static class MockerConnection
    extends ConnectionWrapper {
        private final MockerDataSource dataSource;
        private final int connectionId;
        private boolean ifReuse = false;

        public MockerConnection(MockerDataSource dataSource, Connection connection, int connectionId, boolean ifReuse) throws SQLException {
            super(connection);
            Validate.notNull((Object)dataSource, (String)"DataSource can not be null for MockerConnection");
            if (connection.isClosed() || !connection.isValid(DriverManager.getLoginTimeout())) {
                throw new MockerException("Connection' status is illegal");
            }
            this.connectionId = connectionId;
            this.dataSource = dataSource;
            this.ifReuse = ifReuse;
        }

        public void closeConnection() throws SQLException {
            log.debug("Close Connection from the pool, connectionId={}", (Object)this.connectionId);
            super.close();
        }

        @Override
        public void close() throws SQLException {
            if (!this.ifReuse) {
                super.close();
                return;
            }
            if (this.dataSource.lock.tryLock()) {
                try {
                    Iterator iter = this.dataSource.connectionInUse.iterator();
                    while (iter.hasNext()) {
                        MockerConnection mockerConnection = (MockerConnection)iter.next();
                        if (!mockerConnection.equals(this)) continue;
                        if (mockerConnection.isValid(DriverManager.getLoginTimeout())) {
                            log.debug("Reuse the connection, connectionId={}", (Object)this.connectionId);
                            this.dataSource.connectionPool.push(mockerConnection);
                        } else {
                            super.close();
                        }
                        iter.remove();
                        return;
                    }
                }
                finally {
                    this.dataSource.lock.unlock();
                }
            }
            super.close();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            MockerConnection that = (MockerConnection)o;
            return this.connectionId == that.connectionId;
        }

        public int hashCode() {
            return this.connectionId + "".hashCode();
        }
    }
}

