/*
 * Decompiled with CFR 0.152.
 */
package com.oceanbase.tools.loaddump.common.model.storage;

import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.S3ClientOptions;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.oceanbase.tools.loaddump.common.enums.StorageType;
import com.oceanbase.tools.loaddump.compress.Compressor;
import com.oceanbase.tools.loaddump.utils.LogUtils;
import com.oceanbase.tools.loaddump.utils.StringUtils;
import com.oceanbase.tools.loaddump.vmoption.JavaOpts;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URI;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import lombok.NonNull;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.security.UserGroupInformation;

public class StorageConfig {
    private static final String COS_ENDPOINT_PATTERN = "cos.{0}.myqcloud.com";
    private static final String OBS_ENDPOINT_PATTERN = "obs.{0}.myhuaweicloud.com";
    private static final String OSS_ENDPOINT_PATTERN = "oss-{0}.aliyuncs.com";
    private static final String S3_ENDPOINT_GLOBAL_PATTERN = "s3.{0}.amazonaws.com";
    private static final String S3_ENDPOINT_CN_PATTERN = "s3.{0}.amazonaws.com.cn";
    protected URI uri;
    protected String path;
    protected Configuration fsConf;
    protected StorageType storageType;
    protected FileSystem fs;
    protected Compressor compressor;
    private final Object fsLock = new Object();

    public static StorageConfig create(@NonNull String uri) {
        if (uri == null) {
            throw new NullPointerException("uri is marked non-null but is null");
        }
        StorageConfig config = new StorageConfig();
        config.uri = uri.matches("^[a-zA-Z]:\\\\.*$") ? new File(uri).toURI() : URI.create(uri);
        config.path = config.getUri().getPath();
        config.storageType = StorageType.of(config.uri.getScheme());
        Map<String, String> uriParams = StorageConfig.getUriParams(config.uri.getQuery(), config.storageType);
        Configuration fsConf = new Configuration();
        fsConf.set("fs.defaultFS", "file:///");
        fsConf.set("fs.file.impl", LocalFileSystem.class.getName());
        fsConf.set("fs.hdfs.impl", DistributedFileSystem.class.getName());
        fsConf.set("fs.s3a.impl", S3AFileSystem.class.getName());
        fsConf.set("fs.s3.impl", S3AFileSystem.class.getName());
        fsConf.set("fs.oss.impl", S3AFileSystem.class.getName());
        fsConf.set("fs.cos.impl", S3AFileSystem.class.getName());
        fsConf.set("fs.cosn.impl", S3AFileSystem.class.getName());
        fsConf.set("fs.obs.impl", OBSFileSystem.class.getName());
        if (config.storageType.isObjectStorage()) {
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "access.key", uriParams.get("access-key"));
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "secret.key", uriParams.get("secret-key"));
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "fast.upload.buffer", JavaOpts.uploadBufferType);
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "multipart.size", JavaOpts.uploadBufferSize);
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "fast.upload.active.blocks", JavaOpts.uploadActiveBlocks);
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "attempts.maximum", "3");
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "connection.establish.timeout", "5000");
            fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "connection.maximum", "16384");
        }
        switch (config.storageType) {
            case LOCAL_DISK: {
                break;
            }
            case TENCENT_COS: 
            case ALIYUN_OSS: 
            case AMAZON_S3: 
            case HUAWEI_OBS: {
                String endpoint;
                Preconditions.checkArgument((boolean)StringUtils.isNotBlank(config.uri.getAuthority()), (Object)"Bucket is missing. Check option `-f/--file-path` and try again");
                if (StringUtils.isBlank(config.uri.getPath())) {
                    config.path = File.separator;
                }
                if (StringUtils.isBlank(endpoint = uriParams.get("endpoint"))) {
                    endpoint = StorageConfig.regionToEndpoint(config.storageType, uriParams.get("region"));
                }
                fsConf.set(StorageConfig.getFsConfKeyPrefix(config.storageType) + "endpoint", endpoint);
                break;
            }
            case HADOOP_HDFS: {
                String krb5ConfFile;
                String keytabFile;
                String hdfsSiteFile;
                Preconditions.checkArgument((boolean)StringUtils.isNotBlank(config.uri.getAuthority()), (Object)"Hadoop authority is missing. Check option `-f/--file-path` and try again");
                String coreSiteFile = uriParams.get("core-site-file");
                if (StringUtils.isNotBlank(coreSiteFile)) {
                    Preconditions.checkArgument((boolean)new File(coreSiteFile).exists(), (String)"File: \"%s\" is not exists", (Object)coreSiteFile);
                    fsConf.addResource(new Path(coreSiteFile));
                }
                if (StringUtils.isNotBlank(hdfsSiteFile = uriParams.get("hdfs-site-file"))) {
                    Preconditions.checkArgument((boolean)new File(hdfsSiteFile).exists(), (String)"File: \"%s\" is not exists", (Object)hdfsSiteFile);
                    fsConf.addResource(new Path(hdfsSiteFile));
                }
                if (StringUtils.isNotBlank(keytabFile = uriParams.get("keytab-file"))) {
                    fsConf.set("hadoop.security.authentication", "kerberos");
                }
                if (StringUtils.isNotBlank(krb5ConfFile = uriParams.get("krb5-conf-file"))) {
                    Preconditions.checkArgument((boolean)new File(krb5ConfFile).exists(), (String)"File: \"%s\" is not exists", (Object)krb5ConfFile);
                    System.setProperty("java.security.krb5.conf", krb5ConfFile);
                }
                UserGroupInformation.setConfiguration((Configuration)fsConf);
                if (!StringUtils.isNotBlank(keytabFile)) break;
                Preconditions.checkArgument((boolean)new File(keytabFile).exists(), (String)"File: \"%s\" is not exists", (Object)keytabFile);
                try {
                    UserGroupInformation.loginUserFromKeytab((String)uriParams.get("principal"), (String)keytabFile);
                    break;
                }
                catch (Exception e) {
                    throw new RuntimeException("Login hadoop user failed. Reason: " + e.getMessage());
                }
            }
        }
        config.fsConf = fsConf;
        return config;
    }

    private static Map<String, String> getUriParams(String queryParams, StorageType storageType) {
        if (StringUtils.isBlank(queryParams)) {
            return new HashMap<String, String>();
        }
        Map<String, String> paramMap = Lists.newArrayList((Object[])queryParams.split("&")).stream().collect(Collectors.toMap(e -> StringUtils.substringBefore((String)e, (String)"="), e -> StringUtils.substringAfter((String)e, (String)"=")));
        if (storageType.isObjectStorage()) {
            Preconditions.checkArgument((boolean)paramMap.containsKey("access-key"), (Object)"AccessKey must be specified");
            Preconditions.checkArgument((boolean)paramMap.containsKey("secret-key"), (Object)"SecretKey must be specified");
            Preconditions.checkArgument((paramMap.containsKey("endpoint") || paramMap.containsKey("region") ? 1 : 0) != 0, (Object)"Endpoint or region must be specified");
        }
        return paramMap;
    }

    public FileSystem getFileSystem() {
        if (this.fs != null) {
            return this.fs;
        }
        Object object = this.fsLock;
        synchronized (object) {
            if (this.fs != null) {
                return this.fs;
            }
            try {
                Stopwatch sw = Stopwatch.createStarted();
                this.fs = FileSystem.get((URI)this.uri, (Configuration)this.fsConf);
                this.fs.setVerifyChecksum(false);
                this.fs.setWriteChecksum(false);
                if (JavaOpts.uploadDisableChunkedEncoding.booleanValue() && this.fs instanceof S3AFileSystem) {
                    try {
                        S3AFileSystem s3aFileSystem = (S3AFileSystem)this.fs;
                        Field s3Field = s3aFileSystem.getClass().getDeclaredField("s3");
                        s3Field.setAccessible(true);
                        AmazonS3Client s3 = (AmazonS3Client)s3Field.get(s3aFileSystem);
                        Field s3ImmutableField = s3.getClass().getSuperclass().getDeclaredField("isImmutable");
                        s3ImmutableField.setAccessible(true);
                        s3ImmutableField.setBoolean(s3, false);
                        S3ClientOptions s3ClientOptions = S3ClientOptions.builder().disableChunkedEncoding().build();
                        s3.setS3ClientOptions(s3ClientOptions);
                        s3ImmutableField.setBoolean(s3, true);
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
                if (!this.isLocalFileSystem()) {
                    LogUtils.info("Trying to establish connection to {}.", this.storageType.getName());
                    this.fs.getFileStatus(new Path(this.uri));
                    LogUtils.info("Connecting to {}. Elapsed: {} ms", this.storageType.getName(), sw.elapsed(TimeUnit.MILLISECONDS));
                }
                return this.fs;
            }
            catch (IOException e) {
                throw new RuntimeException("Create file system failed.", e);
            }
        }
    }

    public Path getResourcePath(Path p) {
        if (this.uri.getScheme() == null) {
            return p;
        }
        return new Path(this.uri.getScheme() + "://" + this.uri.getAuthority() + p.toString());
    }

    public boolean isLocalFileSystem() {
        return this.storageType == StorageType.LOCAL_DISK;
    }

    public StorageConfig withLocalTmpPath(String tmpPath) {
        switch (this.storageType) {
            case TENCENT_COS: 
            case ALIYUN_OSS: 
            case AMAZON_S3: 
            case HUAWEI_OBS: {
                this.fsConf.set("fs.s3a.buffer.dir", tmpPath);
                break;
            }
            case HADOOP_HDFS: {
                this.fsConf.set("hadoop.tmp.dir", tmpPath);
                break;
            }
        }
        return this;
    }

    public StorageConfig withCompress(Compressor compressor) {
        this.compressor = compressor;
        return this;
    }

    private static String regionToEndpoint(StorageType storageType, String region) {
        switch (storageType) {
            case TENCENT_COS: {
                return MessageFormat.format(COS_ENDPOINT_PATTERN, region);
            }
            case ALIYUN_OSS: {
                return MessageFormat.format(OSS_ENDPOINT_PATTERN, region);
            }
            case HUAWEI_OBS: {
                return MessageFormat.format(OBS_ENDPOINT_PATTERN, region);
            }
            case AMAZON_S3: {
                if (region.startsWith("cn-")) {
                    return MessageFormat.format(S3_ENDPOINT_CN_PATTERN, region);
                }
                return MessageFormat.format(S3_ENDPOINT_GLOBAL_PATTERN, region);
            }
        }
        throw new IllegalArgumentException("regionToEndpoint is not applicable for storageType " + (Object)((Object)storageType));
    }

    private static String getFsConfKeyPrefix(StorageType type) {
        if (type == StorageType.HUAWEI_OBS) {
            return "fs.obs.";
        }
        if (type.isObjectStorage()) {
            return "fs.s3a.";
        }
        throw new IllegalStateException("You should not get key prefix for non-object-storage type, instead, use full key.");
    }

    public URI getUri() {
        return this.uri;
    }

    public String getPath() {
        return this.path;
    }

    public Configuration getFsConf() {
        return this.fsConf;
    }

    public StorageType getStorageType() {
        return this.storageType;
    }

    public Compressor getCompressor() {
        return this.compressor;
    }
}

