/*
 * Decompiled with CFR 0.152.
 */
package com.oceanbase.tools.loaddump.manager;

import com.alibaba.druid.filter.FilterEventAdapter;
import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.proxy.jdbc.ResultSetProxy;
import com.alibaba.druid.proxy.jdbc.StatementProxy;
import com.google.common.base.Preconditions;
import com.oceanbase.tools.loaddump.common.enums.ServerMode;
import com.oceanbase.tools.loaddump.common.exception.ConnectFailedException;
import com.oceanbase.tools.loaddump.common.model.ConnectionKey;
import com.oceanbase.tools.loaddump.jdbc.JdbcExecutor;
import com.oceanbase.tools.loaddump.utils.DBUtils;
import com.oceanbase.tools.loaddump.utils.ExceptionUtils;
import com.oceanbase.tools.loaddump.utils.JdbcUtils;
import com.oceanbase.tools.loaddump.utils.StringUtils;
import com.oceanbase.tools.loaddump.utils.TimeUtils;
import com.oceanbase.tools.loaddump.vmoption.JavaOpts;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager {
    private static final Logger log = LoggerFactory.getLogger(SessionManager.class);
    private boolean initialized = false;
    private boolean supportSys;
    private ConnectionKey connectionKey;
    private DruidDataSource businessDataSource;
    private DruidDataSource sysDataSource;
    static final String GET_VERSION_OB_COMP_MODE_SQL = "show variables where Variable_name in ('version_comment','ob_compatibility_mode','lower_case_table_names')";

    public SessionManager init(ConnectionKey connectionKey) {
        Preconditions.checkState((!this.initialized ? 1 : 0) != 0, (Object)"A Second initiation is not allowed");
        this.connectionKey = connectionKey;
        log.info("Trying to establish JDBC connection to `{}`...", (Object)connectionKey.getDefaultUser());
        String url = connectionKey.getJdbcUrl4Biz();
        String user = connectionKey.getDefaultUser();
        String password = connectionKey.getPassword();
        try (Connection conn = DriverManager.getConnection(url, user, password);){
            ServerMode serverMode = this.queryServerMode(conn);
            connectionKey.setServerMode(serverMode);
            connectionKey.setDatabase(DBUtils.extractObjectName(connectionKey.getConnectDatabase(), serverMode));
            this.businessDataSource = this.createBusinessDataSource();
            if (connectionKey.getServerMode().isPreviousV4() && !connectionKey.hasNoSysPrivileges()) {
                log.info("Sys-tenant credential is required (add `--no-sys` if you don't have one). Now trying to establish JDBC connection to `{}`...", (Object)connectionKey.getSysTenantUser());
                this.sysDataSource = this.createSystemDataSource();
                this.supportSys = true;
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to establish connection to server, reason: " + e.getMessage() + ".\n- Note:\n-\tTo connect OBServer directly, you may specify `-P/--port` as \"2881\" by default, while `-u/--user` should be of format: '<user>@<tenant>';\n-\tTo connect ODP, you may specify `-P/--port` as \"2883\" by default, while `-u/--user` should be of format: '<user>@<tenant>#<cluster>';\n-\tTo connect Cloud OceanBase, you may add cli arg: `--public-cloud`, while `-u/--user` should be of format: '<user>'.");
        }
        this.initialized = true;
        return this;
    }

    public DruidDataSource getSystemDataSource() throws Exception {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        return this.sysDataSource;
    }

    public DruidDataSource getBusinessDataSource() throws Exception {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        return this.businessDataSource;
    }

    public Connection getPooledBizConnection() throws Exception {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        return this.businessDataSource.getConnection();
    }

    public Connection getPooledSysConnection() throws Exception {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        return this.sysDataSource.getConnection();
    }

    public Connection createNewConnection() {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        int retryTimes = 0;
        while (true) {
            try {
                ++retryTimes;
                String url = this.getConnectionKey().getJdbcUrl4Biz();
                log.debug("Creating new connection. JDBC URL: {}", (Object)url);
                String username = this.getConnectionKey().getDefaultUser();
                String password = this.getConnectionKey().getPassword();
                Connection conn = DriverManager.getConnection(url, username, password);
                List<String> initSqlList = this.connectionKey.getInitSql4Biz();
                if (!initSqlList.isEmpty()) {
                    JdbcUtils.execute(conn, String.join((CharSequence)";", initSqlList));
                }
                return conn;
            }
            catch (Exception ex) {
                boolean shouldRetry;
                String error = ExceptionUtils.getRootCauseMessage(ex);
                boolean bl = shouldRetry = error.contains("Timeout") || error.contains("Connection is closed");
                if (retryTimes < 5 && shouldRetry) {
                    TimeUtils.sleep(TimeUnit.MILLISECONDS, new Random().nextInt(500));
                    log.warn("Retry {} times to get direct connection, ignore it. Cause: {}", (Object)retryTimes, (Object)error);
                    continue;
                }
                throw new ConnectFailedException(ex, "Get direct connection failed. Key: {}", this.getConnectionKey());
            }
            break;
        }
    }

    private DruidDataSource createBusinessDataSource() throws Exception {
        try {
            String url = this.connectionKey.getJdbcUrl4Biz();
            log.debug("JDBC url for business tenant: {}", (Object)url);
            String username = this.connectionKey.getDefaultUser();
            String password = this.connectionKey.getPassword();
            String driverClassName = this.connectionKey.getDriverClassName();
            DruidDataSource dataSource = this.createDataSource(url, username, password, driverClassName);
            dataSource.setConnectionInitSqls(this.connectionKey.getInitSql4Biz());
            dataSource.init();
            return dataSource;
        }
        catch (Exception e) {
            log.error("Create data source for business tenant failed. Error: ", (Throwable)e);
            throw e;
        }
    }

    private DruidDataSource createSystemDataSource() throws Exception {
        try {
            String url = this.connectionKey.getJdbcUrl4Sys();
            log.debug("JDBC url for sys tenant: {}", (Object)url);
            String username = this.connectionKey.getSysTenantUser();
            String password = this.connectionKey.getSysPassword();
            String driverClassName = this.connectionKey.getDriverClassName();
            DruidDataSource dataSource = this.createDataSource(url, username, password, driverClassName);
            dataSource.setConnectionInitSqls(this.connectionKey.getInitSql4Sys());
            dataSource.init();
            return dataSource;
        }
        catch (Exception e) {
            log.error("Create data source for sys tenant failed. Error: ", (Throwable)e);
            throw e;
        }
    }

    private DruidDataSource createDataSource(String url, String user, String password, String driverClassName) {
        DruidDataSource druidDataSource = new DruidDataSource();
        druidDataSource.setUrl(url);
        druidDataSource.setInitialSize(this.connectionKey.getConcurrency());
        druidDataSource.setMinIdle(1);
        druidDataSource.setMaxActive(this.connectionKey.getConcurrency() * 2);
        druidDataSource.setUsername(user);
        druidDataSource.setPassword(password);
        druidDataSource.setMaxWait(180000L);
        druidDataSource.setFailFast(true);
        druidDataSource.setKeepAlive(true);
        druidDataSource.setAsyncInit(true);
        druidDataSource.setTestOnBorrow(true);
        druidDataSource.setTestOnReturn(false);
        druidDataSource.setTestWhileIdle(true);
        druidDataSource.setRemoveAbandoned(true);
        druidDataSource.setMaxCreateTaskCount(8);
        druidDataSource.setNotFullTimeoutRetryCount(-1);
        druidDataSource.setBreakAfterAcquireFailure(true);
        druidDataSource.setValidationQuery("select 1 from dual");
        druidDataSource.setDriverClassName(driverClassName);
        if (JavaOpts.sqlMonitorEnabled.booleanValue()) {
            druidDataSource.getProxyFilters().add(new SqlMonitorAdapter());
        }
        String connectTimeout = this.connectionKey.getSessionConfig().getJdbcOptionByKey("connectTimeout");
        String socketTimeout = this.connectionKey.getSessionConfig().getJdbcOptionByKey("socketTimeout");
        if (StringUtils.isNotEmpty(connectTimeout)) {
            druidDataSource.setConnectTimeout(Integer.parseInt(connectTimeout));
        }
        if (StringUtils.isNotEmpty(socketTimeout)) {
            druidDataSource.setSocketTimeout(Integer.parseInt(socketTimeout));
        }
        return druidDataSource;
    }

    public ServerMode queryServerMode(Connection conn) throws Exception {
        return JdbcExecutor.query(conn, GET_VERSION_OB_COMP_MODE_SQL, rs -> {
            Preconditions.checkState((boolean)rs.first(), (Object)"Query server mode is null");
            rs.beforeFirst();
            ServerMode serverMode = ServerMode.MYSQL;
            String version = null;
            String serverModeStr = null;
            int caseSensitiveMode = 1;
            while (rs.next()) {
                if (StringUtils.equals("version_comment", rs.getString(1))) {
                    String versionComment = rs.getString(2);
                    version = versionComment.split(" ")[1];
                }
                if (StringUtils.equals("ob_compatibility_mode", rs.getString(1))) {
                    serverModeStr = rs.getString(2);
                }
                if (!StringUtils.equals("lower_case_table_names", rs.getString(1))) continue;
                caseSensitiveMode = rs.getInt(2);
            }
            if (StringUtils.isNotBlank(serverModeStr)) {
                serverMode = ServerMode.of(serverModeStr);
            }
            serverMode.setVersion(version);
            serverMode.setCaseSensitive(caseSensitiveMode == 0);
            return serverMode;
        });
    }

    public void destroy() {
        Preconditions.checkState((boolean)this.initialized, (Object)"SessionManager is not initialized");
        Optional.ofNullable(this.businessDataSource).ifPresent(DruidDataSource::close);
        Optional.ofNullable(this.sysDataSource).ifPresent(DruidDataSource::close);
        this.initialized = false;
    }

    public boolean isClosed() {
        return !this.initialized;
    }

    public boolean isSupportSys() {
        return this.supportSys;
    }

    public ConnectionKey getConnectionKey() {
        return this.connectionKey;
    }

    static class SqlMonitorAdapter
    extends FilterEventAdapter {
        static final Logger SQL_MONITOR_LOGGER = LoggerFactory.getLogger((String)"SQLMonitorLogger");

        SqlMonitorAdapter() {
        }

        protected void statementExecuteBefore(StatementProxy statement, String sql) {
            if (this.shouldIgnore(sql)) {
                return;
            }
            super.statementExecuteBefore(statement, sql);
            SQL_MONITOR_LOGGER.info("==> Executing SQL: {}\n", (Object)sql);
        }

        protected void statementExecuteQueryBefore(StatementProxy statement, String sql) {
            super.statementExecuteBefore(statement, sql);
            SQL_MONITOR_LOGGER.info("==> Executing Query SQL: {}\n", (Object)sql);
        }

        protected void statementExecuteUpdateBefore(StatementProxy statement, String sql) {
            super.statementExecuteBefore(statement, sql);
            SQL_MONITOR_LOGGER.info("==> Executing Update SQL: {}\n", (Object)sql);
        }

        protected void statementExecuteAfter(StatementProxy statement, String sql, boolean result) {
            if (this.shouldIgnore(sql)) {
                return;
            }
            super.statementExecuteAfter(statement, sql, result);
            long timeCost = statement.getLastExecuteTimeNano() / 1000000L;
            this.logSlowSqlIfNeeded(sql, timeCost);
        }

        protected void statementExecuteQueryAfter(StatementProxy statement, String sql, ResultSetProxy resultSet) {
            super.statementExecuteQueryAfter(statement, sql, resultSet);
            long timeCost = statement.getLastExecuteTimeNano() / 1000000L;
            this.logSlowSqlIfNeeded(sql, timeCost);
        }

        protected void statementExecuteUpdateAfter(StatementProxy statement, String sql, int updateCount) {
            super.statementExecuteUpdateAfter(statement, sql, updateCount);
            long timeCost = statement.getLastExecuteTimeNano() / 1000000L;
            this.logSlowSqlIfNeeded(sql, timeCost);
        }

        private boolean shouldIgnore(String sql) {
            return "select 1 from dual".equals(sql);
        }

        private void logSlowSqlIfNeeded(String sql, long timeCost) {
            if (timeCost > (long)JavaOpts.slowSqlThreshold) {
                log.warn("[SQL-MONITOR] Slow Query SQL detected (> {}ms): {}. Time taken: {}ms", new Object[]{JavaOpts.slowSqlThreshold, sql, timeCost});
                SQL_MONITOR_LOGGER.warn("Slow Query SQL detected (> {}ms): {}. Time taken: {}ms\n", new Object[]{JavaOpts.slowSqlThreshold, sql, timeCost});
            }
        }
    }
}

