/*
 * Decompiled with CFR 0.152.
 */
package com.openelements.hedera.base.implementation;

import com.hedera.hashgraph.sdk.FileId;
import com.openelements.hedera.base.FileClient;
import com.openelements.hedera.base.HederaException;
import com.openelements.hedera.base.protocol.FileAppendRequest;
import com.openelements.hedera.base.protocol.FileContentsRequest;
import com.openelements.hedera.base.protocol.FileContentsResponse;
import com.openelements.hedera.base.protocol.FileCreateRequest;
import com.openelements.hedera.base.protocol.FileCreateResult;
import com.openelements.hedera.base.protocol.FileDeleteRequest;
import com.openelements.hedera.base.protocol.FileInfoRequest;
import com.openelements.hedera.base.protocol.FileInfoResponse;
import com.openelements.hedera.base.protocol.FileUpdateRequest;
import com.openelements.hedera.base.protocol.ProtocolLayerClient;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileClientImpl
implements FileClient {
    private static final Logger log = LoggerFactory.getLogger(FileClientImpl.class);
    private final ProtocolLayerClient protocolLayerClient;

    public FileClientImpl(@NonNull ProtocolLayerClient protocolLayerClient) {
        this.protocolLayerClient = Objects.requireNonNull(protocolLayerClient, "protocolLevelClient must not be null");
    }

    @Override
    public FileId createFile(@NonNull byte[] contents) throws HederaException {
        return this.createFileImpl(contents, null);
    }

    @Override
    public FileId createFile(@NonNull byte[] contents, @NonNull Instant expirationTime) throws HederaException {
        return this.createFileImpl(contents, expirationTime);
    }

    private FileId createFileImpl(@NonNull byte[] contents, @Nullable Instant expirationTime) throws HederaException {
        Objects.requireNonNull(contents, "fileId must not be null");
        if (contents.length > 1024000) {
            throw new HederaException("File contents must be less than 1024000 bytes");
        }
        if (expirationTime != null && expirationTime.isBefore(Instant.now())) {
            throw new IllegalArgumentException("Expiration time must be in the future");
        }
        if (contents.length <= 2048) {
            FileCreateRequest request = expirationTime != null ? FileCreateRequest.of(contents, expirationTime) : FileCreateRequest.of(contents);
            FileCreateResult result = this.protocolLayerClient.executeFileCreateTransaction(request);
            return result.fileId();
        }
        if (log.isDebugEnabled()) {
            int appendCount = Math.floorDiv(contents.length, 2048);
            log.debug("Content of size {} is to big for 1 FileCreate transaction. Will append {} FileAppend transactions", (Object)contents.length, (Object)appendCount);
        }
        byte[] start = Arrays.copyOf(contents, 2048);
        FileCreateRequest request = FileCreateRequest.of(start);
        FileCreateResult result = this.protocolLayerClient.executeFileCreateTransaction(request);
        FileId fileId = result.fileId();
        byte[] remaining = Arrays.copyOfRange(contents, 2048, contents.length);
        while (remaining.length > 0) {
            int length = Math.min(remaining.length, 2048);
            byte[] next = Arrays.copyOf(remaining, length);
            FileAppendRequest appendRequest = FileAppendRequest.of(fileId, next);
            this.protocolLayerClient.executeFileAppendRequestTransaction(appendRequest);
            remaining = Arrays.copyOfRange(remaining, length, remaining.length);
        }
        return fileId;
    }

    @Override
    @NonNull
    public byte[] readFile(@NonNull FileId fileId) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        try {
            FileContentsRequest request = FileContentsRequest.of(fileId);
            FileContentsResponse response = this.protocolLayerClient.executeFileContentsQuery(request);
            return response.contents();
        }
        catch (Exception e) {
            throw new HederaException("Failed to read file with fileId " + String.valueOf(fileId), e);
        }
    }

    @Override
    public void deleteFile(@NonNull FileId fileId) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        try {
            FileDeleteRequest request = FileDeleteRequest.of(fileId);
            this.protocolLayerClient.executeFileDeleteTransaction(request);
        }
        catch (Exception e) {
            throw new HederaException("Failed to delete file with fileId " + String.valueOf(fileId), e);
        }
    }

    @Override
    public void updateFile(@NonNull FileId fileId, @NonNull byte[] content) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        Objects.requireNonNull(content, "content must not be null");
        if (content.length > 1024000) {
            throw new HederaException("File contents must be less than 1024000 bytes");
        }
        if (content.length <= 2048) {
            FileUpdateRequest request = FileUpdateRequest.of(fileId, content);
            this.protocolLayerClient.executeFileUpdateRequestTransaction(request);
        } else {
            if (log.isDebugEnabled()) {
                int appendCount = Math.floorDiv(content.length, 2048);
                log.debug("Content of size {} is to big for 1 FileUpdate transaction. Will append {} FileAppend transactions", (Object)content.length, (Object)appendCount);
            }
            byte[] start = Arrays.copyOf(content, 2048);
            FileUpdateRequest request = FileUpdateRequest.of(fileId, start);
            this.protocolLayerClient.executeFileUpdateRequestTransaction(request);
            byte[] remaining = Arrays.copyOfRange(content, 2048, content.length);
            while (remaining.length > 0) {
                int length = Math.min(remaining.length, 2048);
                byte[] next = Arrays.copyOf(remaining, length);
                FileAppendRequest appendRequest = FileAppendRequest.of(fileId, next);
                this.protocolLayerClient.executeFileAppendRequestTransaction(appendRequest);
                remaining = Arrays.copyOfRange(remaining, length, remaining.length);
            }
        }
    }

    @Override
    public void updateExpirationTime(@NonNull FileId fileId, @NonNull Instant expirationTime) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        Objects.requireNonNull(expirationTime, "expirationTime must not be null");
        if (expirationTime.isBefore(Instant.now())) {
            throw new IllegalArgumentException("Expiration time must be in the future");
        }
        FileUpdateRequest request = FileUpdateRequest.of(fileId, expirationTime);
        this.protocolLayerClient.executeFileUpdateRequestTransaction(request);
    }

    @Override
    public boolean isDeleted(@NonNull FileId fileId) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        FileInfoRequest request = FileInfoRequest.of(fileId);
        FileInfoResponse infoResponse = this.protocolLayerClient.executeFileInfoQuery(request);
        return infoResponse.deleted();
    }

    @Override
    public int getSize(@NonNull FileId fileId) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        FileInfoRequest request = FileInfoRequest.of(fileId);
        FileInfoResponse infoResponse = this.protocolLayerClient.executeFileInfoQuery(request);
        return infoResponse.size();
    }

    @Override
    public Instant getExpirationTime(@NonNull FileId fileId) throws HederaException {
        Objects.requireNonNull(fileId, "fileId must not be null");
        FileInfoRequest request = FileInfoRequest.of(fileId);
        FileInfoResponse infoResponse = this.protocolLayerClient.executeFileInfoQuery(request);
        return infoResponse.expirationTime();
    }
}

