package com.licorize.api.client;

import java.util.List;

/**
 * @author Federico Soldani - fsoldani@open-lab.com
 */
public interface LicorizeClient extends LicorizeAuthenticationClient {

  /**
   * Returns strips from time line in descending order
   * @return strips list
   */
  LicorizeApiResponse getStripList();

  /**
   * Returns the last strips from the user time line
   * @param parameters: filtering parameters
   *      Optional:
   *          type - Strip type e.g. 'BOOKMARK'
   *          types - Strips type list comma separated
   *          team - Project Id
   *          user - User Id - If not specified it's the current user
   *          searchText - Text filter
   *          tags - Tag list comma separated
   *          millis - timeline millis (see lastMillis on the response)
   *          limitResultTo - number of results - default is 10
   * @return strip list
   *          more - true if there are more strips in result list
   *          totStrips - the whole number of matching strips
   *          lastMillis - the oldest timeStamp in the response (used in next call if more is true on "millis" parameter)
   *          strips - strip list
   *          tags - an array of tags used by matching strips
   */
  LicorizeApiResponse getStripList(List<Parameter<String, String>> parameters);

  /**
   * Returns the strip identified by the specified ID.
   * @param stripId
   * @return strip
   */
  LicorizeApiResponse getStrip(int stripId);

  /**
   * Update single strip
   * @param stripId
   * @param parameters
   *      Optional:
   *          type - Strip Type e.g. 'BOOKMARK'
   *          teamId - Project Id
   *          notes - Strip Notes
   *          ownerId - User Id
   *          title - Strip title
   *          url - Strip url
   *          email - Strip email
   *          tags - Strip tags (comma separated list)
   *          sticky - Is Sticky Strip - ('yes' - 'no')
   *          worklogDone - total worklog done
   *          dueDate - Strip due date
   *          time - if present 'dueDate' - for instance "REMAINDER"
   *          done - ('yes' - 'no') - Strip done
   * @return strip updated
   */
  LicorizeApiResponse updateStrip(int stripId, List<Parameter<String, String>> parameters);

  /**
   * Create single strip
   * @param parameters
   *      Optional:
   *          type - Strip Type e.g. 'BOOKMARK'
   *          teamId - Project Id
   *          notes - Strip Notes
   *          ownerId - User Id
   *          title - Strip title
   *          url - Strip url
   *          email - Strip email
   *          tags - Strip tags (comma separated list)
   *          sticky - Is Sticky Strip - ('yes' - 'no')
   *          worklogDone - total worklog done
   *          dueDate - Strip dueDate
   *          time - if present 'dueDate' - for instance "REMAINDER"
   *          done - ('yes' - 'no') - Strip done
   * @return strip created
   */
  LicorizeApiResponse createStrip(List<Parameter<String, String>> parameters);

  /**
   * Remove the strip identified by the specified ID.
   * @param stripId
   * @return success or fail operation
   */
  LicorizeApiResponse removeStrip(int stripId);

  /**
   * Change a strip type.
   * @param stripId
   * @param stripType
   * @return strip updated
   */
  LicorizeApiResponse changeStripType(int stripId, String stripType);

  /**
   * Return the projects list for the current user.
   * @return projects list
   */
  LicorizeApiResponse getProjectsList();

  /**
   * Update project.
   * @param projectId
   * @param parameters
   *      Optional:
   *          name - Team name
   *          description - Team description
   *          isActive - ('yes' - 'no')
   *          isVisible - ('yes' - 'no')
   *          code - Team code for public name
   * @return project updated
   */
  LicorizeApiResponse updateProject(int projectId, List<Parameter<String, String>> parameters);

  /**
   * Create project
   * @param parameters
   *      Optional:
   *          name - Team name
   *          description - Team description
   *          isActive - ('yes' - 'no')
   *          isVisible - ('yes' - 'no')
   *          code - Team code for public name
   * @return project created
   */
  LicorizeApiResponse createProject(List<Parameter<String, String>> parameters);

  /**
   * Return the project identified by the specified ID.
   * @param projectId
   * @return project
   */
  LicorizeApiResponse getProject(int projectId);

  /**
   * Return the project members list.
   * @param projectId
   * @return project members
   */
  LicorizeApiResponse getProjectMembers(int projectId);

  /**
   * Remove single project identified by the specified ID.
   * @param projectId
   * @return success or fail operation
   */  
  LicorizeApiResponse removeProject(int projectId);

  /**
   * Show the complete tag cloud.
   * @return tags list for current user
   */
  LicorizeApiResponse getTagsList();

  /**
   * Get the current user.
   * @return current user
   */
  LicorizeApiResponse getUser();

  /**
   * Get user public data from userName.
   * @param userName
   * @return public user
   */
  LicorizeApiResponse getPublicUser(String userName);

  /**
   * Method for saving a new remind me later strip.
   * @param url The url that will be saved.
   * @return The created strip
   */
  LicorizeApiResponse remindMeLater(String url);

  /**
   * Method for save a new remind me later strip.
   * @param url The url that will be saved.
   * @param parameters
   *        title - Bookmark title
   *        notes - Bookmark Notes
   *        tags  - Bookmark Tags - Default is "remindMeLater"
   * @return The created strip
   */
  LicorizeApiResponse remindMeLater(String url, List<Parameter<String, String>> parameters);

}