package com.licorize.api.client;

import com.licorize.api.client.constant.LicorizeClientConstants;
import org.scribe.model.Token;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.util.Map;
import java.util.concurrent.*;

/**
 * A factory class for LicorizeClient
 *
 * @author Federico Soldani - fsoldani@open-lab.com
 */
public class LicorizeClientFactory implements Serializable {
  /**
   * The Constant factoriesMap.
   */
  private static final Map<Token, LicorizeClientFactory> factoriesMap = new ConcurrentHashMap<Token, LicorizeClientFactory>();

  private static final long serialVersionUID = -8786350214116968924L;

  /**
   * The api consumer.
   */
  private Token apiConsumer;

  /**
   * The default client impl.
   */
  private Constructor<? extends LicorizeClient> defaultClientImpl;

  /**
   * Instantiates a new linked in api client factory.
   *
   * @param apiConsumer the api consumer
   */
  private LicorizeClientFactory(Token apiConsumer) {
    this.apiConsumer = apiConsumer;
  }

  /**
   * New instance.
   *
   * @param consumerKey    the consumer key
   * @param consumerSecret the consumer secret
   * @return the linked in api client factory
   */
  public static LicorizeClientFactory newInstance(String consumerKey, String consumerSecret) {
    return newInstance(new Token(consumerKey, consumerSecret));
  }

  /**
   * New instance.
   *
   * @param apiConsumer the api consumer
   * @return the linked in api client factory
   */
  public static synchronized LicorizeClientFactory newInstance(Token apiConsumer) {
    validateConsumerKey(apiConsumer);

    LicorizeClientFactory factory = factoriesMap.get(apiConsumer);

    if (factory == null) {
      factory = new LicorizeClientFactory(apiConsumer);
      factoriesMap.put(apiConsumer, factory);
    }

    return factory;
  }
  // factory methods for OAuth support


  /**
   * Creates a new LicorizeClient object.
   *
   * @return the Licorize client
   */
  public LicorizeClient createLicorizeClient() {
    try {
      if (defaultClientImpl == null) {
        Class<? extends LicorizeClient> clazz = (Class<? extends LicorizeClient>) Class.forName(LicorizeClientConstants.CLIENT_DEFAULT_IMPL);

        defaultClientImpl = clazz.getConstructor(String.class, String.class);
      }

      final LicorizeClient client = defaultClientImpl.newInstance(apiConsumer.getToken(), apiConsumer.getSecret());

      return client;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  /**
   * Creates a new LicorizeClient object.
   *
   * @return the Licorize client
   */
  public LicorizeClient createLicorizeClient(Class<? extends LicorizeClient> implClass) {
    try {
      final LicorizeClient client = implClass.getConstructor(String.class, String.class).newInstance(apiConsumer.getToken(), apiConsumer.getSecret());

      return client;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  /**
   * Creates a new LicorizeClient object.
   *
   * @param accessToken the access token
   * @return the Licorize client
   */
  public LicorizeClient createLicorizeClient(Token accessToken) {
    validateAccessToken(accessToken);
    try {
      if (defaultClientImpl == null) {
        Class<? extends LicorizeClient> clazz = (Class<? extends LicorizeClient>) Class.forName(LicorizeClientConstants.CLIENT_DEFAULT_IMPL);

        defaultClientImpl = clazz.getConstructor(String.class, String.class);
      }

      final LicorizeClient client = defaultClientImpl.newInstance(apiConsumer.getToken(), apiConsumer.getSecret());

      client.setAccessToken(accessToken);

      return client;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  /**
   * Creates a new LicorizeClient object.
   *
   * @param accessToken the access token
   * @return the Licorize client
   */
  public LicorizeClient createLicorizeClient(Class<? extends LicorizeClient> implClass, Token accessToken) {
    validateAccessToken(accessToken);
    try {
      final LicorizeClient client = implClass.getConstructor(String.class, String.class).newInstance(apiConsumer.getToken(), apiConsumer.getSecret());

      client.setAccessToken(accessToken);

      return client;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  private static void validateConsumerKey(Token apiConsumer) {
    if (apiConsumer == null) {
      throw new IllegalArgumentException("api consumer cannot be null.");
    }
    if (apiConsumer.getToken() == null || apiConsumer.getToken().length() == 0) {
      throw new IllegalArgumentException("consumer key cannot be null or empty.");
    }
    if (apiConsumer.getSecret() == null || apiConsumer.getSecret().length() == 0) {
      throw new IllegalArgumentException("consumer secret cannot be null or empty.");
    }
  }

  private void validateAccessToken(Token accessToken) {
    if (accessToken == null) {
      throw new IllegalArgumentException("access token cannot be null.");
    }
    if (accessToken.getToken() == null || accessToken.getToken().length() == 0) {
      throw new IllegalArgumentException("access token cannot be null or empty.");
    }
    if (accessToken.getSecret() == null || accessToken.getSecret().length() == 0) {
      throw new IllegalArgumentException("access token secret cannot be null or empty.");
    }
  }
}
