package com.licorize.api.client.constant;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Federico Soldani - fsoldani@open-lab.com
 */
public final class LicorizeApiUrls {

  /**
   * The Constant API_URLS_FILE.
   */
  public static final String API_URLS_FILE = "LicorizeApiUrls.properties";

  /**
   * The static logger.
   */
  private static final Logger LOG = Logger.getLogger(LicorizeApiUrls.class.getCanonicalName());

  /**
   * The Constant linkedInApiUrls.
   */
  private static final Properties licorizeApiUrls = new Properties();

  static {
    try {
      licorizeApiUrls.load(LicorizeApiUrls.class.getResourceAsStream(API_URLS_FILE));
    } catch (IOException e) {
      LOG.log(Level.SEVERE, "An error occurred while loading urls.", e);
    }
  }

  /**
   * The Constant GET_STRIPS_LIST.
   */
  public static final String GET_STRIPS_LIST =
          licorizeApiUrls.getProperty("com.licorize.api.client.getStripsList");

  /**
   * The Constant GET_STRIP.
   */
  public static final String GET_STRIP =
          licorizeApiUrls.getProperty("com.licorize.api.client.getStrip");

  /**
   * The Constant UPDATE_STRIP.
   */
  public static final String UPDATE_STRIP =
          licorizeApiUrls.getProperty("com.licorize.api.client.updateStrip");

  /**
   * The Constant REMOVE_STRIP.
   */
  public static final String REMOVE_STRIP =
          licorizeApiUrls.getProperty("com.licorize.api.client.removeStrip");

  /**
   * The Constant CHANGE_STRIP_TYPE.
   */
  public static final String CHANGE_STRIP_TYPE =
          licorizeApiUrls.getProperty("com.licorize.api.client.changeStripType");

  /**
   * The Constant GET_PROJECTS_LIST.
   */
  public static final String GET_PROJECTS_LIST =
          licorizeApiUrls.getProperty("com.licorize.api.client.getProjectsList");

  /**
   * The Constant UPDATE_PROJECT.
   */
  public static final String UPDATE_PROJECT =
          licorizeApiUrls.getProperty("com.licorize.api.client.updateProject");

  /**
   * The Constant GET_PROJECT.
   */
  public static final String GET_PROJECT =
          licorizeApiUrls.getProperty("com.licorize.api.client.getProject");

  /**
   * The Constant GET_PROJECT_MEMBERS.
   */
  public static final String GET_PROJECT_MEMBERS =
          licorizeApiUrls.getProperty("com.licorize.api.client.getProjectMembers");

  /**
   * The Constant REMOVE_PROJECT.
   */
  public static final String REMOVE_PROJECT =
          licorizeApiUrls.getProperty("com.licorize.api.client.removeProject");

  /**
   * The Constant GET_TAGS_LIST.
   */
  public static final String GET_TAGS_LIST =
          licorizeApiUrls.getProperty("com.licorize.api.client.getTagsList");

  /**
   * The Constant GET_USER.
   */
  public static final String GET_USER =
          licorizeApiUrls.getProperty("com.licorize.api.client.getUser");

  /**
   * The Constant GET_PUBLIC_USER.
   */
  public static final String GET_PUBLIC_USER =
          licorizeApiUrls.getProperty("com.licorize.api.client.getPublicUser");

  /**
   * The Constant REMIND_ME_LATER.
   */
  public static final String REMIND_ME_LATER =
          licorizeApiUrls.getProperty("com.licorize.api.client.remindMeLater");

  /**
   * The Constant LICORIZE_OAUTH_REQUEST_TOKEN_URL.
   */
  public static final String LICORIZE_OAUTH_REQUEST_TOKEN_URL =
          licorizeApiUrls.getProperty("com.licorize.api.client.oauth.requestToken");

  /**
   * The Constant LICORIZE_OAUTH_AUTHORIZE_URL.
   */
  public static final String LICORIZE_OAUTH_AUTHORIZE_URL =
          licorizeApiUrls.getProperty("com.licorize.api.client.oauth.authorize");

  /**
   * The Constant LICORIZE_OAUTH_AUTHORIZE_CALL_URL.
   */
  public static final String LICORIZE_OAUTH_AUTHORIZE_CALL_URL =
          licorizeApiUrls.getProperty("com.licorize.api.client.oauth.authorizeUrl");

  /**
   * The Constant LICORIZE_OAUTH_ACCESS_TOKEN_URL.
   */
  public static final String LICORIZE_OAUTH_ACCESS_TOKEN_URL =
          licorizeApiUrls.getProperty("com.licorize.api.client.oauth.accessToken");

  /**
   * The Constant LICORIZE_OAUTH_INVALIDATE_TOKEN_URL.
   */
  public static final String LICORIZE_OAUTH_INVALIDATE_TOKEN_URL =
          licorizeApiUrls.getProperty("com.licorize.api.client.oauth.invalidateToken");

  /**
   * Instantiates a new linked in api urls.
   */
  private LicorizeApiUrls() {
  }

  /**
   * The Class LicorizeApiUrlBuilder.
   */
  public static class LicorizeApiUrlBuilder {

    /**
     * The Constant API_URLS_PLACEHOLDER_START.
     */
    private static final char API_URLS_PLACEHOLDER_START = '{';

    /**
     * The Constant API_URLS_PLACEHOLDER_END.
     */
    private static final char API_URLS_PLACEHOLDER_END = '}';

    /**
     * The Constant QUERY_PARAMETERS_PLACEHOLDER.
     */
    private static final String QUERY_PARAMETERS_PLACEHOLDER = "queryParameters";

    /**
     * The url format.
     */
    private String urlFormat;

    /**
     * The parameters map.
     */
    private Map<String, Collection<String>> parametersMap = new HashMap<String, Collection<String>>();

    /**
     * The fields map.
     */
    private Map<String, String> fieldsMap = new HashMap<String, String>();

    /**
     * Instantiates a new linked in api url builder.
     *
     * @param urlFormat the url format
     */
    public LicorizeApiUrlBuilder(String urlFormat) {
      this.urlFormat = urlFormat;
    }

    /**
     * With parameter.
     *
     * @param name  the name
     * @param value the value
     * @return the linked in api url builder
     */
    public LicorizeApiUrlBuilder withParameter(String name, String value) {
      if (value != null && value.length() > 0) {
        parametersMap.put(name, Collections.singleton(encodeUrl(value)));
      }

      return this;
    }

    /**
     * With parameters.
     *
     * @param name   the name
     * @param values the values
     * @return the linked in api url builder
     */
    public LicorizeApiUrlBuilder withParameters(String name, Collection<String> values) {
      List<String> encodedValues = new ArrayList<String>(values.size());
      for (String value : values) {
        encodedValues.add(encodeUrl(value));
      }
      parametersMap.put(name, encodedValues);

      return this;
    }

    /**
     * With empty field.
     *
     * @param name the name
     * @return the linked in api url builder
     */
    public LicorizeApiUrlBuilder withEmptyField(String name) {
      fieldsMap.put(name, "");

      return this;
    }

    /**
     * With field.
     *
     * @param name  the name
     * @param value the value
     * @return the linked in api url builder
     */
    public LicorizeApiUrlBuilder withField(String name, String value) {
      withField(name, value, false);

      return this;
    }

    /**
     * With field.
     *
     * @param name  the name
     * @param value the value
     * @return the linked in api url builder
     */
    public LicorizeApiUrlBuilder withField(String name, String value, boolean escape) {
      if (escape) {
        fieldsMap.put(name, encodeUrl(value));
      } else {
        fieldsMap.put(name, value);
      }

      return this;
    }


    /**
     * Builds the url.
     *
     * @return the string
     */
    public String buildUrl() {
      StringBuilder urlBuilder = new StringBuilder();
      StringBuilder placeHolderBuilder = new StringBuilder();
      boolean placeHolderFlag = false;
      for (int i = 0; i < urlFormat.length(); i++) {
        if (urlFormat.charAt(i) == API_URLS_PLACEHOLDER_START) {
          placeHolderBuilder = new StringBuilder();
          placeHolderFlag = true;
        } else if (placeHolderFlag && urlFormat.charAt(i) == API_URLS_PLACEHOLDER_END) {
          String placeHolder = placeHolderBuilder.toString();
          if (fieldsMap.containsKey(placeHolder)) {
            urlBuilder.append(fieldsMap.get(placeHolder));
          } else if (QUERY_PARAMETERS_PLACEHOLDER.equals(placeHolder)) {
            StringBuilder builder = new StringBuilder();
            if (!parametersMap.isEmpty()) {
              builder.append("?");
              Iterator<String> iter = parametersMap.keySet().iterator();
              while (iter.hasNext()) {
                String name = iter.next();
                Collection<String> parameterValues = parametersMap.get(name);
                Iterator<String> iterParam = parameterValues.iterator();
                while (iterParam.hasNext()) {
                  builder.append(name);
                  builder.append("=");
                  builder.append(iterParam.next());
                  if (iterParam.hasNext()) {
                    builder.append("&");
                  }
                }
                if (iter.hasNext()) {
                  builder.append("&");
                }
              }
            }
            urlBuilder.append(builder.toString());
          } else {
            // we did not find a binding for the placeholder.
            // append it as it is.
            urlBuilder.append(API_URLS_PLACEHOLDER_START);
            urlBuilder.append(placeHolder);
            urlBuilder.append(API_URLS_PLACEHOLDER_END);
          }
          placeHolderFlag = false;
        } else if (placeHolderFlag) {
          placeHolderBuilder.append(urlFormat.charAt(i));
        } else {
          urlBuilder.append(urlFormat.charAt(i));
        }
      }

      return urlBuilder.toString();
    }

    /**
     * @param original
     * @return
     */
    private static String encodeUrl(String original) {
      try {
        return URLEncoder.encode(original, LicorizeClientConstants.CONTENT_ENCODING);
      } catch (UnsupportedEncodingException e) {
        // should never be here..
        return original;
      }
    }
  }
}
