package com.licorize.api.client.impl;

import com.licorize.api.client.*;
import com.licorize.api.client.constant.LicorizeApiUrls;
import com.licorize.api.client.constant.LicorizeClientConstants;
import com.licorize.api.client.oauth.LicorizeApi;
import net.sf.json.JSONObject;
import org.scribe.builder.ServiceBuilder;
import org.scribe.model.*;
import org.scribe.oauth.OAuthService;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

/**
 * @author Federico Soldani - fsoldani@open-lab.com
 */
public class LicorizeClientImpl implements LicorizeClient {

  /**
   * The static logger.
   */
  protected final Logger LOG = Logger.getLogger(getClass().getCanonicalName());

  private Token accessToken;

  private Token apiConsumer;

  /**
   * Constructs ...
   *
   * @param consumerKey
   * @param consumerSecret
   */
  public LicorizeClientImpl(String consumerKey, String consumerSecret) {
    this.apiConsumer = new Token(consumerKey, consumerSecret);
  }

  /* ****************************************** */
  /* ****************** OAUTH ***************** */
  /* ****************************************** */
  
  @Override
  public Token getRequestToken() {
    try {
      OAuthService service = this.buildService();

      return service.getRequestToken();
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  @Override
  public Token getAccessToken(Token requestToken) {
    validateRequestToken(requestToken);
    try {
      OAuthService service = this.buildService();

      this.accessToken = service.getAccessToken(requestToken, new Verifier(""));

      return this.accessToken;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  @Override
  public String getAuthenticationUrl(Token requestToken) {
    validateRequestToken(requestToken);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.LICORIZE_OAUTH_AUTHORIZE_CALL_URL);

    return builder.withParameter("oauth_token", requestToken.getToken()).buildUrl();
  }

  @Override
  public void setAccessToken(Token accessToken) {
    this.accessToken = accessToken;
  }

  /* ****************************************** */
  /* **************** STRIP LIST ************** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getStripList() {
    return this.getStripList(null);
  }

  @Override
  public LicorizeApiResponse getStripList(List<Parameter<String, String>> parameters) {
    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return readResponse(callOAuthApiMethod(LicorizeApiUrls.GET_STRIPS_LIST, parameters));
  }

  /* ****************************************** */
  /* **************** SHOW STRIP ************** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getStrip(int stripId) {
    assertPositiveNumber("strip ID", stripId);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.GET_STRIP);
    String apiUrl = builder.withField("STRIP_ID", String.valueOf(stripId)).buildUrl();

    return readResponse(callOAuthApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* *************** UPDATE STRIP ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse updateStrip(int stripId, List<Parameter<String, String>> parameters) {
    assertPositiveNumber("strip ID", stripId);

    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    _parameters.add(new Parameter("id", stripId));
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return this.createStrip(_parameters);
  }

  /* ****************************************** */
  /* *************** CREATE STRIP ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse createStrip(List<Parameter<String, String>> parameters) {
    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return readResponse(callOAuthApiMethod(LicorizeApiUrls.UPDATE_STRIP, _parameters));
  }

  /* ****************************************** */
  /* *************** REMOVE STRIP ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse removeStrip(int stripId) {
    assertPositiveNumber("strip ID", stripId);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.REMOVE_STRIP);
    String apiUrl = builder.withField("STRIP_ID", String.valueOf(stripId)).buildUrl();

    return readResponse(callOAuthApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* ************ CHANGE STRIP TYPE *********** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse changeStripType(int stripId, String stripType) {
    assertPositiveNumber("strip ID", stripId);
    assertNotNullOrEmpty("strip type", stripType);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.CHANGE_STRIP_TYPE);
    String apiUrl = builder.withField("STRIP_ID", String.valueOf(stripId)).buildUrl();

    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    _parameters.add(new Parameter("type", stripType));
    

    return readResponse(callOAuthApiMethod(apiUrl, _parameters));
  }

  /* ****************************************** */
  /* ************* GET PROJECTS LIST ********** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getProjectsList() {
    return readResponse(callOAuthApiMethod(LicorizeApiUrls.GET_PROJECTS_LIST));
  }

  /* ****************************************** */
  /* *************** UPDATE PROJECT *********** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse updateProject(int projectId, List<Parameter<String, String>> parameters) {
    assertPositiveNumber("project ID", projectId);

    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    _parameters.add(new Parameter("id", projectId));
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return this.createProject(_parameters);
  }

  /* ****************************************** */
  /* ************** CREATE PROJECT ************ */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse createProject(List<Parameter<String, String>> parameters) {
    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return readResponse(callOAuthApiMethod(LicorizeApiUrls.UPDATE_PROJECT, _parameters));
  }

  /* ****************************************** */
  /* *************** SHOW PROJECT ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getProject(int projectId) {
    assertPositiveNumber("project ID", projectId);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.GET_PROJECT);
    String apiUrl = builder.withField("PROJECT_ID", String.valueOf(projectId)).buildUrl();

    return readResponse(callOAuthApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* *********** GET PROJECT MEMBERS ********** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getProjectMembers(int projectId) {
    assertPositiveNumber("project ID", projectId);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.GET_PROJECT_MEMBERS);
    String apiUrl = builder.withField("PROJECT_ID", String.valueOf(projectId)).buildUrl();

    return readResponse(callOAuthApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* ************* REMOVE PROJECT ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse removeProject(int projectId) {
    assertPositiveNumber("project ID", projectId);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.REMOVE_PROJECT);
    String apiUrl = builder.withField("PROJECT_ID", String.valueOf(projectId)).buildUrl();

    return readResponse(callOAuthApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* ************** GET TAGS LIST ************* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getTagsList() {
    return readResponse(callOAuthApiMethod(LicorizeApiUrls.GET_TAGS_LIST));
  }

  /* ****************************************** */
  /* **************** SHOW USER *************** */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getUser() {
    return readResponse(callOAuthApiMethod(LicorizeApiUrls.GET_USER));
  }

  /* ****************************************** */
  /* ********* SHOW USER PUBLIC PROFILE ******* */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse getPublicUser(String userName) {
    assertNotNullOrEmpty("user name", userName);

    LicorizeApiUrls.LicorizeApiUrlBuilder builder = createLicorizeApiUrlBuilder(LicorizeApiUrls.GET_PUBLIC_USER);
    String apiUrl = builder.withField("USER_NAME", String.valueOf(userName)).buildUrl();

    return readResponse(callApiMethod(apiUrl));
  }

  /* ****************************************** */
  /* ************* REMIND ME LATER ************ */
  /* ****************************************** */

  @Override
  public LicorizeApiResponse remindMeLater(String url) {
    return this.remindMeLater(url, null);
  }

  @Override
  public LicorizeApiResponse remindMeLater(String url, List<Parameter<String, String>> parameters) {
    assertNotNullOrEmpty("url", url);

    List<Parameter<String, String>> _parameters = new ArrayList<Parameter<String, String>>();
    _parameters.add(new Parameter("url", url));
    if (parameters != null) {
      _parameters.addAll(parameters);
    }

    return readResponse(callOAuthApiMethod(LicorizeApiUrls.REMIND_ME_LATER, _parameters));
  }

  /* ****************************************** */

  protected LicorizeApiUrls.LicorizeApiUrlBuilder createLicorizeApiUrlBuilder(String urlFormat) {
    return new LicorizeApiUrls.LicorizeApiUrlBuilder(urlFormat);
  }

  protected Response callOAuthApiMethod(String apiUrl) {
    final List<Parameter<String, String>> parameters = Collections.emptyList();
    return this.callOAuthApiMethod(apiUrl, parameters, Verb.POST);
  }

  protected Response callOAuthApiMethod(String apiUrl, List<Parameter<String, String>> parameters) {
    return this.callOAuthApiMethod(apiUrl, parameters, Verb.POST);
  }

  protected Response callOAuthApiMethod(String apiUrl, List<Parameter<String, String>> parameters, Verb verb) {
    try {
      OAuthService service = this.buildService();

      OAuthRequest request = new OAuthRequest(verb, apiUrl);

      if (null != parameters) {
        for (Parameter parameter : parameters) {
          request.addBodyParameter(String.valueOf(parameter.getName()), String.valueOf(parameter.getValue()));
        }
      }

      service.signRequest(accessToken, request);

      //request.setConnectTimeout(LicorizeClientConstants.CONNECT_TIMEOUT, TimeUnit.MILLISECONDS);
      request.setTimeout(LicorizeClientConstants.CONNECT_TIMEOUT);

      return request.send();
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  protected Response callApiMethod(String apiUrl) {
    return this.callApiMethod(apiUrl, Verb.GET);
  }

  protected Response callApiMethod(String apiUrl, Verb verb) {
    try {
      OAuthRequest request = new OAuthRequest(verb, apiUrl);

      request.setTimeout(LicorizeClientConstants.CONNECT_TIMEOUT);

      return request.send();
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  protected LicorizeApiResponse readResponse(Response response) {
    return unmarshallObject(response);
  }

  protected LicorizeApiResponse unmarshallObject(Response response) {
    try {
      String responseBody = response.getBody();
      JSONObject responseJson = JSONObject.fromObject(responseBody);

      LicorizeApiResponseImpl apiResponse = new LicorizeApiResponseImpl();
      apiResponse.setSuccess(responseJson.getBoolean("ok"));
      apiResponse.setError(responseJson.optString("error"));
      apiResponse.setMessage(responseJson.optString("message"));
      apiResponse.setResponseString(responseBody);
      apiResponse.setResponseObject(responseJson);

      return apiResponse;
    } catch (Exception e) {
      throw new LicorizeClientException(e);
    }
  }

  protected OAuthService buildService() {
    OAuthService service = new ServiceBuilder()
            .provider(LicorizeApi.class)
            .apiKey(apiConsumer.getToken())
            .apiSecret(apiConsumer.getSecret())
            .build();

    return service;
  }

  /**
   * @param s
   * @return
   */
  protected boolean isNullOrEmpty(String s) {
    return ((s == null) || (s.length() == 0));
  }

  /**
   * @param name
   * @param value
   */
  protected void assertNotNullOrEmpty(String name, String value) {
    if (isNullOrEmpty(value)) {
      throw new IllegalArgumentException(name + " cannot be null or empty.");
    }
  }

  /**
   * @param name
   * @param value
   */
  protected void assertPositiveNumber(String name, int value) {
    if (value < 0) {
      throw new IllegalArgumentException(name + " cannot be less than zero.");
    }
  }

  private static void validateRequestToken(Token requestToken) {
    if (requestToken == null) {
      throw new IllegalArgumentException("request token cannot be null.");
    }
    if (requestToken.getToken() == null || requestToken.getToken().length() == 0) {
      throw new IllegalArgumentException("request token cannot be null or empty.");
    }
  }


}