/*
 * Copyright (C) 2021 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.capfloor;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.google.common.collect.ImmutableMap;
import com.opengamma.strata.basics.currency.CurrencyAmount;
import com.opengamma.strata.product.capfloor.IborCapletFloorletPeriod;

/**
 * A map of currency amounts keyed by Ibor caplet/floorlet periods.
 * <p>
 * The currency amounts should represent the same metric.
 */
@BeanDefinition
public final class IborCapletFloorletPeriodCurrencyAmounts implements ImmutableBean, Serializable {

  /**
   * The map of Ibor caplet/floorlet periods to the currency amount.
   */
  @PropertyDefinition(validate = "notNull")
  private final ImmutableMap<IborCapletFloorletPeriod, CurrencyAmount> amounts;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance of currency amounts.
   *
   * @param currencyAmountMap the map of currency amounts
   * @return the instance
   */
  public static IborCapletFloorletPeriodCurrencyAmounts of(
      Map<IborCapletFloorletPeriod, CurrencyAmount> currencyAmountMap) {

    return new IborCapletFloorletPeriodCurrencyAmounts(currencyAmountMap);
  }

  /**
   * Gets a currency amount for the provided Ibor caplet/floorlet.
   *
   * @param period the ibor caplet/floorlet
   * @return the currency amount, empty if missing
   */
  public Optional<CurrencyAmount> findAmount(IborCapletFloorletPeriod period) {
    return Optional.ofNullable(amounts.get(period));
  }

  /**
   * Gets a currency amount for the provided Ibor caplet/floorlet.
   *
   * @param period the ibor caplet/floorlet
   * @return the currency amount
   * @throws IllegalArgumentException if the period is missing
   */
  public CurrencyAmount getAmount(IborCapletFloorletPeriod period) {
    if (!amounts.containsKey(period)) {
      throw new IllegalArgumentException("Could not find currency amount for " + period);
    }
    return amounts.get(period);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code IborCapletFloorletPeriodCurrencyAmounts}.
   * @return the meta-bean, not null
   */
  public static IborCapletFloorletPeriodCurrencyAmounts.Meta meta() {
    return IborCapletFloorletPeriodCurrencyAmounts.Meta.INSTANCE;
  }

  static {
    MetaBean.register(IborCapletFloorletPeriodCurrencyAmounts.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static IborCapletFloorletPeriodCurrencyAmounts.Builder builder() {
    return new IborCapletFloorletPeriodCurrencyAmounts.Builder();
  }

  private IborCapletFloorletPeriodCurrencyAmounts(
      Map<IborCapletFloorletPeriod, CurrencyAmount> amounts) {
    JodaBeanUtils.notNull(amounts, "amounts");
    this.amounts = ImmutableMap.copyOf(amounts);
  }

  @Override
  public IborCapletFloorletPeriodCurrencyAmounts.Meta metaBean() {
    return IborCapletFloorletPeriodCurrencyAmounts.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the map of Ibor caplet/floorlet periods to the currency amount.
   * @return the value of the property, not null
   */
  public ImmutableMap<IborCapletFloorletPeriod, CurrencyAmount> getAmounts() {
    return amounts;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      IborCapletFloorletPeriodCurrencyAmounts other = (IborCapletFloorletPeriodCurrencyAmounts) obj;
      return JodaBeanUtils.equal(amounts, other.amounts);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(amounts);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("IborCapletFloorletPeriodCurrencyAmounts{");
    buf.append("amounts").append('=').append(JodaBeanUtils.toString(amounts));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code IborCapletFloorletPeriodCurrencyAmounts}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code amounts} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<ImmutableMap<IborCapletFloorletPeriod, CurrencyAmount>> amounts = DirectMetaProperty.ofImmutable(
        this, "amounts", IborCapletFloorletPeriodCurrencyAmounts.class, (Class) ImmutableMap.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "amounts");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -879772901:  // amounts
          return amounts;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public IborCapletFloorletPeriodCurrencyAmounts.Builder builder() {
      return new IborCapletFloorletPeriodCurrencyAmounts.Builder();
    }

    @Override
    public Class<? extends IborCapletFloorletPeriodCurrencyAmounts> beanType() {
      return IborCapletFloorletPeriodCurrencyAmounts.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code amounts} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ImmutableMap<IborCapletFloorletPeriod, CurrencyAmount>> amounts() {
      return amounts;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -879772901:  // amounts
          return ((IborCapletFloorletPeriodCurrencyAmounts) bean).getAmounts();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code IborCapletFloorletPeriodCurrencyAmounts}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<IborCapletFloorletPeriodCurrencyAmounts> {

    private Map<IborCapletFloorletPeriod, CurrencyAmount> amounts = ImmutableMap.of();

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(IborCapletFloorletPeriodCurrencyAmounts beanToCopy) {
      this.amounts = beanToCopy.getAmounts();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -879772901:  // amounts
          return amounts;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -879772901:  // amounts
          this.amounts = (Map<IborCapletFloorletPeriod, CurrencyAmount>) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public IborCapletFloorletPeriodCurrencyAmounts build() {
      return new IborCapletFloorletPeriodCurrencyAmounts(
          amounts);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the map of Ibor caplet/floorlet periods to the currency amount.
     * @param amounts  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder amounts(Map<IborCapletFloorletPeriod, CurrencyAmount> amounts) {
      JodaBeanUtils.notNull(amounts, "amounts");
      this.amounts = amounts;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("IborCapletFloorletPeriodCurrencyAmounts.Builder{");
      buf.append("amounts").append('=').append(JodaBeanUtils.toString(amounts));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
