/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.credit;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableValidator;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.StandardId;
import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.data.MarketDataName;
import com.opengamma.strata.market.param.ParameterMetadata;
import com.opengamma.strata.market.param.ParameterPerturbation;

/**
 * The constant recovery rate.
 * <p>
 * The recovery rate is constant for any given date.
 */
@BeanDefinition(builderScope = "private")
public final class ConstantRecoveryRates
    implements RecoveryRates, ImmutableBean, Serializable {

  /**
   * The legal entity identifier.
   * <p>
   * This identifier is used for the reference legal entity of a credit derivative.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final StandardId legalEntityId;
  /**
   * The valuation date.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final LocalDate valuationDate;
  /**
   * The recovery rate.
   * <p>
   * The recovery rate is represented in decimal form, and must be between 0 and 1 inclusive.
   */
  @PropertyDefinition
  private final double recoveryRate;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance.
   * 
   * @param legalEntityId  the legal entity identifier
   * @param valuationDate  the valuation date
   * @param recoveryRate  the recovery rate
   * @return the instance
   */
  public static ConstantRecoveryRates of(StandardId legalEntityId, LocalDate valuationDate, double recoveryRate) {
    return new ConstantRecoveryRates(legalEntityId, valuationDate, recoveryRate);
  }

  @ImmutableValidator
  private void validate() {
    ArgChecker.inRangeInclusive(recoveryRate, 0d, 1d, "recoveryRate");
  }

  //-------------------------------------------------------------------------
  @Override
  public double recoveryRate(LocalDate date) {
    return recoveryRate;
  }

  @Override
  public <T> Optional<T> findData(MarketDataName<T> name) {
    return Optional.empty();
  }

  @Override
  public int getParameterCount() {
    return 1;
  }

  @Override
  public double getParameter(int parameterIndex) {
    ArgChecker.isTrue(parameterIndex == 0, "Only one parameter for ConstantRecoveryRates");
    return recoveryRate;
  }

  @Override
  public ParameterMetadata getParameterMetadata(int parameterIndex) {
    return ParameterMetadata.empty();
  }

  @Override
  public ConstantRecoveryRates withParameter(int parameterIndex, double newValue) {
    ArgChecker.isTrue(parameterIndex == 0, "Only one parameter for ConstantRecoveryRates");
    return new ConstantRecoveryRates(legalEntityId, valuationDate, newValue);
  }

  @Override
  public ConstantRecoveryRates withPerturbation(ParameterPerturbation perturbation) {
    double perturbedValue = perturbation.perturbParameter(0, recoveryRate, getParameterMetadata(0));
    return new ConstantRecoveryRates(legalEntityId, valuationDate, perturbedValue);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ConstantRecoveryRates}.
   * @return the meta-bean, not null
   */
  public static ConstantRecoveryRates.Meta meta() {
    return ConstantRecoveryRates.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ConstantRecoveryRates.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private ConstantRecoveryRates(
      StandardId legalEntityId,
      LocalDate valuationDate,
      double recoveryRate) {
    JodaBeanUtils.notNull(legalEntityId, "legalEntityId");
    JodaBeanUtils.notNull(valuationDate, "valuationDate");
    this.legalEntityId = legalEntityId;
    this.valuationDate = valuationDate;
    this.recoveryRate = recoveryRate;
    validate();
  }

  @Override
  public ConstantRecoveryRates.Meta metaBean() {
    return ConstantRecoveryRates.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the legal entity identifier.
   * <p>
   * This identifier is used for the reference legal entity of a credit derivative.
   * @return the value of the property, not null
   */
  @Override
  public StandardId getLegalEntityId() {
    return legalEntityId;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the valuation date.
   * @return the value of the property, not null
   */
  @Override
  public LocalDate getValuationDate() {
    return valuationDate;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the recovery rate.
   * <p>
   * The recovery rate is represented in decimal form, and must be between 0 and 1 inclusive.
   * @return the value of the property
   */
  public double getRecoveryRate() {
    return recoveryRate;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ConstantRecoveryRates other = (ConstantRecoveryRates) obj;
      return JodaBeanUtils.equal(legalEntityId, other.legalEntityId) &&
          JodaBeanUtils.equal(valuationDate, other.valuationDate) &&
          JodaBeanUtils.equal(recoveryRate, other.recoveryRate);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(legalEntityId);
    hash = hash * 31 + JodaBeanUtils.hashCode(valuationDate);
    hash = hash * 31 + JodaBeanUtils.hashCode(recoveryRate);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(128);
    buf.append("ConstantRecoveryRates{");
    buf.append("legalEntityId").append('=').append(JodaBeanUtils.toString(legalEntityId)).append(',').append(' ');
    buf.append("valuationDate").append('=').append(JodaBeanUtils.toString(valuationDate)).append(',').append(' ');
    buf.append("recoveryRate").append('=').append(JodaBeanUtils.toString(recoveryRate));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ConstantRecoveryRates}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code legalEntityId} property.
     */
    private final MetaProperty<StandardId> legalEntityId = DirectMetaProperty.ofImmutable(
        this, "legalEntityId", ConstantRecoveryRates.class, StandardId.class);
    /**
     * The meta-property for the {@code valuationDate} property.
     */
    private final MetaProperty<LocalDate> valuationDate = DirectMetaProperty.ofImmutable(
        this, "valuationDate", ConstantRecoveryRates.class, LocalDate.class);
    /**
     * The meta-property for the {@code recoveryRate} property.
     */
    private final MetaProperty<Double> recoveryRate = DirectMetaProperty.ofImmutable(
        this, "recoveryRate", ConstantRecoveryRates.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "legalEntityId",
        "valuationDate",
        "recoveryRate");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return legalEntityId;
        case 113107279:  // valuationDate
          return valuationDate;
        case 2002873877:  // recoveryRate
          return recoveryRate;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends ConstantRecoveryRates> builder() {
      return new ConstantRecoveryRates.Builder();
    }

    @Override
    public Class<? extends ConstantRecoveryRates> beanType() {
      return ConstantRecoveryRates.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code legalEntityId} property.
     * @return the meta-property, not null
     */
    public MetaProperty<StandardId> legalEntityId() {
      return legalEntityId;
    }

    /**
     * The meta-property for the {@code valuationDate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LocalDate> valuationDate() {
      return valuationDate;
    }

    /**
     * The meta-property for the {@code recoveryRate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> recoveryRate() {
      return recoveryRate;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return ((ConstantRecoveryRates) bean).getLegalEntityId();
        case 113107279:  // valuationDate
          return ((ConstantRecoveryRates) bean).getValuationDate();
        case 2002873877:  // recoveryRate
          return ((ConstantRecoveryRates) bean).getRecoveryRate();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ConstantRecoveryRates}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<ConstantRecoveryRates> {

    private StandardId legalEntityId;
    private LocalDate valuationDate;
    private double recoveryRate;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return legalEntityId;
        case 113107279:  // valuationDate
          return valuationDate;
        case 2002873877:  // recoveryRate
          return recoveryRate;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          this.legalEntityId = (StandardId) newValue;
          break;
        case 113107279:  // valuationDate
          this.valuationDate = (LocalDate) newValue;
          break;
        case 2002873877:  // recoveryRate
          this.recoveryRate = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public ConstantRecoveryRates build() {
      return new ConstantRecoveryRates(
          legalEntityId,
          valuationDate,
          recoveryRate);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(128);
      buf.append("ConstantRecoveryRates.Builder{");
      buf.append("legalEntityId").append('=').append(JodaBeanUtils.toString(legalEntityId)).append(',').append(' ');
      buf.append("valuationDate").append('=').append(JodaBeanUtils.toString(valuationDate)).append(',').append(' ');
      buf.append("recoveryRate").append('=').append(JodaBeanUtils.toString(recoveryRate));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
