/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.impl.tree;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.product.common.PutCall;

/**
 * European vanilla option function.
 */
@BeanDefinition(builderScope = "private")
public final class EuropeanVanillaOptionFunction
    implements OptionFunction, ImmutableBean, Serializable {

  /**
   * The strike value.
   */
  @PropertyDefinition
  private final double strike;
  /**
   * The time to expiry.
   */
  @PropertyDefinition(overrideGet = true)
  private final double timeToExpiry;
  /**
   * The sign.
   * <p>
   * The sign is +1 for call and -1 for put.
   */
  @PropertyDefinition
  private final double sign;

  /**
   * The number of time steps.
   */
  @PropertyDefinition(overrideGet = true)
  private final int numberOfSteps;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance.
   * 
   * @param strike  the strike
   * @param timeToExpiry  the time to expiry
   * @param putCall  put or call
   * @param numberOfSteps  the number of time steps
   * @return the instance
   */
  public static EuropeanVanillaOptionFunction of(double strike, double timeToExpiry, PutCall putCall, int numberOfSteps) {
    double sign = putCall.isCall() ? 1d : -1d;
    ArgChecker.isTrue(numberOfSteps > 0, "the number of steps should be positive");
    return new EuropeanVanillaOptionFunction(strike, timeToExpiry, sign, numberOfSteps);
  }

  //-------------------------------------------------------------------------
  @Override
  public DoubleArray getPayoffAtExpiryTrinomial(DoubleArray stateValue) {
    int nNodes = stateValue.size();
    double[] values = new double[nNodes];
    for (int i = 0; i < nNodes; ++i) {
      values[i] = Math.max(sign * (stateValue.get(i) - strike), 0d);
    }
    return DoubleArray.ofUnsafe(values);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code EuropeanVanillaOptionFunction}.
   * @return the meta-bean, not null
   */
  public static EuropeanVanillaOptionFunction.Meta meta() {
    return EuropeanVanillaOptionFunction.Meta.INSTANCE;
  }

  static {
    MetaBean.register(EuropeanVanillaOptionFunction.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private EuropeanVanillaOptionFunction(
      double strike,
      double timeToExpiry,
      double sign,
      int numberOfSteps) {
    this.strike = strike;
    this.timeToExpiry = timeToExpiry;
    this.sign = sign;
    this.numberOfSteps = numberOfSteps;
  }

  @Override
  public EuropeanVanillaOptionFunction.Meta metaBean() {
    return EuropeanVanillaOptionFunction.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the strike value.
   * @return the value of the property
   */
  public double getStrike() {
    return strike;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the time to expiry.
   * @return the value of the property
   */
  @Override
  public double getTimeToExpiry() {
    return timeToExpiry;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the sign.
   * <p>
   * The sign is +1 for call and -1 for put.
   * @return the value of the property
   */
  public double getSign() {
    return sign;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the number of time steps.
   * @return the value of the property
   */
  @Override
  public int getNumberOfSteps() {
    return numberOfSteps;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      EuropeanVanillaOptionFunction other = (EuropeanVanillaOptionFunction) obj;
      return JodaBeanUtils.equal(strike, other.strike) &&
          JodaBeanUtils.equal(timeToExpiry, other.timeToExpiry) &&
          JodaBeanUtils.equal(sign, other.sign) &&
          (numberOfSteps == other.numberOfSteps);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(strike);
    hash = hash * 31 + JodaBeanUtils.hashCode(timeToExpiry);
    hash = hash * 31 + JodaBeanUtils.hashCode(sign);
    hash = hash * 31 + JodaBeanUtils.hashCode(numberOfSteps);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("EuropeanVanillaOptionFunction{");
    buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
    buf.append("timeToExpiry").append('=').append(JodaBeanUtils.toString(timeToExpiry)).append(',').append(' ');
    buf.append("sign").append('=').append(JodaBeanUtils.toString(sign)).append(',').append(' ');
    buf.append("numberOfSteps").append('=').append(JodaBeanUtils.toString(numberOfSteps));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code EuropeanVanillaOptionFunction}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code strike} property.
     */
    private final MetaProperty<Double> strike = DirectMetaProperty.ofImmutable(
        this, "strike", EuropeanVanillaOptionFunction.class, Double.TYPE);
    /**
     * The meta-property for the {@code timeToExpiry} property.
     */
    private final MetaProperty<Double> timeToExpiry = DirectMetaProperty.ofImmutable(
        this, "timeToExpiry", EuropeanVanillaOptionFunction.class, Double.TYPE);
    /**
     * The meta-property for the {@code sign} property.
     */
    private final MetaProperty<Double> sign = DirectMetaProperty.ofImmutable(
        this, "sign", EuropeanVanillaOptionFunction.class, Double.TYPE);
    /**
     * The meta-property for the {@code numberOfSteps} property.
     */
    private final MetaProperty<Integer> numberOfSteps = DirectMetaProperty.ofImmutable(
        this, "numberOfSteps", EuropeanVanillaOptionFunction.class, Integer.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "strike",
        "timeToExpiry",
        "sign",
        "numberOfSteps");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -891985998:  // strike
          return strike;
        case -1831499397:  // timeToExpiry
          return timeToExpiry;
        case 3530173:  // sign
          return sign;
        case -1323103225:  // numberOfSteps
          return numberOfSteps;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends EuropeanVanillaOptionFunction> builder() {
      return new EuropeanVanillaOptionFunction.Builder();
    }

    @Override
    public Class<? extends EuropeanVanillaOptionFunction> beanType() {
      return EuropeanVanillaOptionFunction.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code strike} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> strike() {
      return strike;
    }

    /**
     * The meta-property for the {@code timeToExpiry} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> timeToExpiry() {
      return timeToExpiry;
    }

    /**
     * The meta-property for the {@code sign} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> sign() {
      return sign;
    }

    /**
     * The meta-property for the {@code numberOfSteps} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Integer> numberOfSteps() {
      return numberOfSteps;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -891985998:  // strike
          return ((EuropeanVanillaOptionFunction) bean).getStrike();
        case -1831499397:  // timeToExpiry
          return ((EuropeanVanillaOptionFunction) bean).getTimeToExpiry();
        case 3530173:  // sign
          return ((EuropeanVanillaOptionFunction) bean).getSign();
        case -1323103225:  // numberOfSteps
          return ((EuropeanVanillaOptionFunction) bean).getNumberOfSteps();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code EuropeanVanillaOptionFunction}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<EuropeanVanillaOptionFunction> {

    private double strike;
    private double timeToExpiry;
    private double sign;
    private int numberOfSteps;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -891985998:  // strike
          return strike;
        case -1831499397:  // timeToExpiry
          return timeToExpiry;
        case 3530173:  // sign
          return sign;
        case -1323103225:  // numberOfSteps
          return numberOfSteps;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -891985998:  // strike
          this.strike = (Double) newValue;
          break;
        case -1831499397:  // timeToExpiry
          this.timeToExpiry = (Double) newValue;
          break;
        case 3530173:  // sign
          this.sign = (Double) newValue;
          break;
        case -1323103225:  // numberOfSteps
          this.numberOfSteps = (Integer) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public EuropeanVanillaOptionFunction build() {
      return new EuropeanVanillaOptionFunction(
          strike,
          timeToExpiry,
          sign,
          numberOfSteps);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("EuropeanVanillaOptionFunction.Builder{");
      buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
      buf.append("timeToExpiry").append('=').append(JodaBeanUtils.toString(timeToExpiry)).append(',').append(' ');
      buf.append("sign").append('=').append(JodaBeanUtils.toString(sign)).append(',').append(' ');
      buf.append("numberOfSteps").append('=').append(JodaBeanUtils.toString(numberOfSteps));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
