/*
 * Copyright (C) 2023 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.index;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.data.MarketDataName;
import com.opengamma.strata.data.NamedMarketDataId;

/**
 * An identifier used to access Overnight future option volatilities by name.
 * <p>
 * This is used when there is a need to obtain an instance of {@link OvernightFutureOptionVolatilities}.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class OvernightFutureOptionVolatilitiesId
    implements NamedMarketDataId<OvernightFutureOptionVolatilities>, ImmutableBean, Serializable {

  /**
   * The name of the volatilities.
   */
  @PropertyDefinition(validate = "notNull")
  private final OvernightFutureOptionVolatilitiesName name;

  //-------------------------------------------------------------------------
  /**
   * Obtains an identifier used to find Overnight future option volatilities.
   *
   * @param name  the name
   * @return an identifier for the volatilities
   */
  public static OvernightFutureOptionVolatilitiesId of(String name) {
    return new OvernightFutureOptionVolatilitiesId(OvernightFutureOptionVolatilitiesName.of(name));
  }

  /**
   * Obtains an identifier used to find Overnight future option volatilities.
   *
   * @param name  the name
   * @return an identifier for the volatilities
   */
  public static OvernightFutureOptionVolatilitiesId of(OvernightFutureOptionVolatilitiesName name) {
    return new OvernightFutureOptionVolatilitiesId(name);
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<OvernightFutureOptionVolatilities> getMarketDataType() {
    return OvernightFutureOptionVolatilities.class;
  }

  @Override
  public MarketDataName<OvernightFutureOptionVolatilities> getMarketDataName() {
    return name;
  }

  @Override
  public String toString() {
    return "OvernightFutureOptionVolatilitiesId:" + name;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code OvernightFutureOptionVolatilitiesId}.
   */
  private static final TypedMetaBean<OvernightFutureOptionVolatilitiesId> META_BEAN =
      LightMetaBean.of(
          OvernightFutureOptionVolatilitiesId.class,
          MethodHandles.lookup(),
          new String[] {
              "name"},
          new Object[0]);

  /**
   * The meta-bean for {@code OvernightFutureOptionVolatilitiesId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<OvernightFutureOptionVolatilitiesId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private OvernightFutureOptionVolatilitiesId(
      OvernightFutureOptionVolatilitiesName name) {
    JodaBeanUtils.notNull(name, "name");
    this.name = name;
  }

  @Override
  public TypedMetaBean<OvernightFutureOptionVolatilitiesId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the name of the volatilities.
   * @return the value of the property, not null
   */
  public OvernightFutureOptionVolatilitiesName getName() {
    return name;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      OvernightFutureOptionVolatilitiesId other = (OvernightFutureOptionVolatilitiesId) obj;
      return JodaBeanUtils.equal(name, other.name);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(name);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
