/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.swaption;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutablePreBuild;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.collect.tuple.Pair;
import com.opengamma.strata.market.param.ParameterMetadata;

/**
 * Surface node metadata for a surface node for swaptions with a specific time to expiry and simple moneyness.
 * <p>
 * This typically represents a node of swaption volatility surface parameterized by expiry and simple moneyness.
 */
@BeanDefinition(builderScope = "private")
public final class SwaptionSurfaceExpirySimpleMoneynessParameterMetadata
    implements ParameterMetadata, ImmutableBean, Serializable {

  /**
  * The year fraction of the surface node.
  * <p>
  * This is the time to expiry that the node on the surface is defined as.
  * There is not necessarily a direct relationship with a date from an underlying instrument.
  */
  @PropertyDefinition
  private final double yearFraction;
  /**
  * The simple moneyness of the surface node.
  * <p>
  * This is defined as strike - forward.
  */
  @PropertyDefinition
  private final double simpleMoneyness;
  /**
   * The label that describes the node.
   */
  @PropertyDefinition(validate = "notEmpty", overrideGet = true)
  private final String label;

  //-------------------------------------------------------------------------
  /**
   * Creates node metadata using swap convention, year fraction and simple moneyness.
   * 
   * @param yearFraction  the year fraction
   * @param simpleMoneyness  the simple moneyness
   * @return node metadata 
   */
  public static SwaptionSurfaceExpirySimpleMoneynessParameterMetadata of(
      double yearFraction,
      double simpleMoneyness) {

    String label = Pair.of(yearFraction, simpleMoneyness).toString();
    return new SwaptionSurfaceExpirySimpleMoneynessParameterMetadata(yearFraction, simpleMoneyness, label);
  }

  /**
   * Creates node using swap convention, year fraction, simple moneyness and label.
   * 
   * @param yearFraction  the year fraction
   * @param simpleMoneyness  the simple moneyness
   * @param label  the label to use
   * @return the metadata
   */
  public static SwaptionSurfaceExpirySimpleMoneynessParameterMetadata of(
      double yearFraction,
      double simpleMoneyness,
      String label) {

    return new SwaptionSurfaceExpirySimpleMoneynessParameterMetadata(yearFraction, simpleMoneyness, label);
  }

  @ImmutablePreBuild
  private static void preBuild(Builder builder) {
    if (builder.label == null) {
      builder.label = Pair.of(builder.yearFraction, builder.simpleMoneyness).toString();
    }
  }

  @Override
  public Pair<Double, Double> getIdentifier() {
    return Pair.of(yearFraction, simpleMoneyness);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SwaptionSurfaceExpirySimpleMoneynessParameterMetadata}.
   * @return the meta-bean, not null
   */
  public static SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Meta meta() {
    return SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private SwaptionSurfaceExpirySimpleMoneynessParameterMetadata(
      double yearFraction,
      double simpleMoneyness,
      String label) {
    JodaBeanUtils.notEmpty(label, "label");
    this.yearFraction = yearFraction;
    this.simpleMoneyness = simpleMoneyness;
    this.label = label;
  }

  @Override
  public SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Meta metaBean() {
    return SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the year fraction of the surface node.
   * <p>
   * This is the time to expiry that the node on the surface is defined as.
   * There is not necessarily a direct relationship with a date from an underlying instrument.
   * @return the value of the property
   */
  public double getYearFraction() {
    return yearFraction;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the simple moneyness of the surface node.
   * <p>
   * This is defined as strike - forward.
   * @return the value of the property
   */
  public double getSimpleMoneyness() {
    return simpleMoneyness;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the label that describes the node.
   * @return the value of the property, not empty
   */
  @Override
  public String getLabel() {
    return label;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SwaptionSurfaceExpirySimpleMoneynessParameterMetadata other = (SwaptionSurfaceExpirySimpleMoneynessParameterMetadata) obj;
      return JodaBeanUtils.equal(yearFraction, other.yearFraction) &&
          JodaBeanUtils.equal(simpleMoneyness, other.simpleMoneyness) &&
          JodaBeanUtils.equal(label, other.label);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(yearFraction);
    hash = hash * 31 + JodaBeanUtils.hashCode(simpleMoneyness);
    hash = hash * 31 + JodaBeanUtils.hashCode(label);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(128);
    buf.append("SwaptionSurfaceExpirySimpleMoneynessParameterMetadata{");
    buf.append("yearFraction").append('=').append(JodaBeanUtils.toString(yearFraction)).append(',').append(' ');
    buf.append("simpleMoneyness").append('=').append(JodaBeanUtils.toString(simpleMoneyness)).append(',').append(' ');
    buf.append("label").append('=').append(JodaBeanUtils.toString(label));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SwaptionSurfaceExpirySimpleMoneynessParameterMetadata}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code yearFraction} property.
     */
    private final MetaProperty<Double> yearFraction = DirectMetaProperty.ofImmutable(
        this, "yearFraction", SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.class, Double.TYPE);
    /**
     * The meta-property for the {@code simpleMoneyness} property.
     */
    private final MetaProperty<Double> simpleMoneyness = DirectMetaProperty.ofImmutable(
        this, "simpleMoneyness", SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.class, Double.TYPE);
    /**
     * The meta-property for the {@code label} property.
     */
    private final MetaProperty<String> label = DirectMetaProperty.ofImmutable(
        this, "label", SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.class, String.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "yearFraction",
        "simpleMoneyness",
        "label");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return yearFraction;
        case 802881925:  // simpleMoneyness
          return simpleMoneyness;
        case 102727412:  // label
          return label;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends SwaptionSurfaceExpirySimpleMoneynessParameterMetadata> builder() {
      return new SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Builder();
    }

    @Override
    public Class<? extends SwaptionSurfaceExpirySimpleMoneynessParameterMetadata> beanType() {
      return SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code yearFraction} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> yearFraction() {
      return yearFraction;
    }

    /**
     * The meta-property for the {@code simpleMoneyness} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> simpleMoneyness() {
      return simpleMoneyness;
    }

    /**
     * The meta-property for the {@code label} property.
     * @return the meta-property, not null
     */
    public MetaProperty<String> label() {
      return label;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return ((SwaptionSurfaceExpirySimpleMoneynessParameterMetadata) bean).getYearFraction();
        case 802881925:  // simpleMoneyness
          return ((SwaptionSurfaceExpirySimpleMoneynessParameterMetadata) bean).getSimpleMoneyness();
        case 102727412:  // label
          return ((SwaptionSurfaceExpirySimpleMoneynessParameterMetadata) bean).getLabel();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SwaptionSurfaceExpirySimpleMoneynessParameterMetadata}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<SwaptionSurfaceExpirySimpleMoneynessParameterMetadata> {

    private double yearFraction;
    private double simpleMoneyness;
    private String label;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return yearFraction;
        case 802881925:  // simpleMoneyness
          return simpleMoneyness;
        case 102727412:  // label
          return label;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          this.yearFraction = (Double) newValue;
          break;
        case 802881925:  // simpleMoneyness
          this.simpleMoneyness = (Double) newValue;
          break;
        case 102727412:  // label
          this.label = (String) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public SwaptionSurfaceExpirySimpleMoneynessParameterMetadata build() {
      preBuild(this);
      return new SwaptionSurfaceExpirySimpleMoneynessParameterMetadata(
          yearFraction,
          simpleMoneyness,
          label);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(128);
      buf.append("SwaptionSurfaceExpirySimpleMoneynessParameterMetadata.Builder{");
      buf.append("yearFraction").append('=').append(JodaBeanUtils.toString(yearFraction)).append(',').append(' ');
      buf.append("simpleMoneyness").append('=').append(JodaBeanUtils.toString(simpleMoneyness)).append(',').append(' ');
      buf.append("label").append('=').append(JodaBeanUtils.toString(label));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
