/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.payment;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.basics.currency.Payment;
import com.opengamma.strata.product.ResolvedProduct;

/**
 * A bullet payment, resolved for pricing.
 * <p>
 * This is the resolved form of {@link BulletPayment} and is an input to the pricers.
 * Applications will typically create a {@code ResolvedBulletPayment} from a {@code BulletPayment}
 * using {@link BulletPayment#resolve(ReferenceData)}.
 * <p>
 * A {@code ResolvedBulletPayment} is bound to data that changes over time, such as holiday calendars.
 * If the data changes, such as the addition of a new holiday, the resolved form will not be updated.
 * Care must be taken when placing the resolved form in a cache or persistence layer.
 */
@BeanDefinition
public final class ResolvedBulletPayment
    implements ResolvedProduct, ImmutableBean, Serializable {

  /**
   * The payment to be made.
   */
  @PropertyDefinition(validate = "notNull")
  private final Payment payment;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance of a resolved bullet payment.
   * 
   * @param payment  the payment
   * @return the resolved bullet payment
   */
  public static ResolvedBulletPayment of(Payment payment) {
    return new ResolvedBulletPayment(payment);
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the currency of this payment.
   * 
   * @return the payment currency
   */
  public Currency getCurrency() {
    return payment.getCurrency();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ResolvedBulletPayment}.
   * @return the meta-bean, not null
   */
  public static ResolvedBulletPayment.Meta meta() {
    return ResolvedBulletPayment.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ResolvedBulletPayment.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static ResolvedBulletPayment.Builder builder() {
    return new ResolvedBulletPayment.Builder();
  }

  private ResolvedBulletPayment(
      Payment payment) {
    JodaBeanUtils.notNull(payment, "payment");
    this.payment = payment;
  }

  @Override
  public ResolvedBulletPayment.Meta metaBean() {
    return ResolvedBulletPayment.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the payment to be made.
   * @return the value of the property, not null
   */
  public Payment getPayment() {
    return payment;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ResolvedBulletPayment other = (ResolvedBulletPayment) obj;
      return JodaBeanUtils.equal(payment, other.payment);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(payment);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("ResolvedBulletPayment{");
    buf.append("payment").append('=').append(JodaBeanUtils.toString(payment));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ResolvedBulletPayment}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code payment} property.
     */
    private final MetaProperty<Payment> payment = DirectMetaProperty.ofImmutable(
        this, "payment", ResolvedBulletPayment.class, Payment.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "payment");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return payment;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public ResolvedBulletPayment.Builder builder() {
      return new ResolvedBulletPayment.Builder();
    }

    @Override
    public Class<? extends ResolvedBulletPayment> beanType() {
      return ResolvedBulletPayment.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code payment} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Payment> payment() {
      return payment;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return ((ResolvedBulletPayment) bean).getPayment();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ResolvedBulletPayment}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<ResolvedBulletPayment> {

    private Payment payment;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(ResolvedBulletPayment beanToCopy) {
      this.payment = beanToCopy.getPayment();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return payment;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          this.payment = (Payment) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public ResolvedBulletPayment build() {
      return new ResolvedBulletPayment(
          payment);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the payment to be made.
     * @param payment  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder payment(Payment payment) {
      JodaBeanUtils.notNull(payment, "payment");
      this.payment = payment;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("ResolvedBulletPayment.Builder{");
      buf.append("payment").append('=').append(JodaBeanUtils.toString(payment));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
