/*
 * Copyright (C) 2018 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.rate;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableConstructor;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableSet;
import com.opengamma.strata.basics.index.Index;

/**
 * Defines a known annual fixed rate of interest that follows overnight compounding.
 * <p>
 * An interest rate that is specified in the contract.
 * This is typically used in the fixed legs of a Brazilian swap.
 */
@BeanDefinition(builderScope = "private")
public final class FixedOvernightCompoundedAnnualRateComputation
    implements RateComputation, ImmutableBean, Serializable {

  /**
   * The fixed rate for overnight compounding.
   * A 5% rate will be expressed as 0.05.
   */
  @PropertyDefinition
  private final double rate;
  /**
   * The accrual factor.
   */
  @PropertyDefinition
  private final double accrualFactor;
  /**
   * The calculated simple rate.
   */
  private final double simpleRate;  // not a property

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from the rate and accrual factor.
   * 
   * @param rate  the fixed rate
   * @param accrualFactor  the accrual factor
   * @return the fixed rate computation
   */
  public static FixedOvernightCompoundedAnnualRateComputation of(double rate, double accrualFactor) {
    return new FixedOvernightCompoundedAnnualRateComputation(rate, accrualFactor);
  }

  //-------------------------------------------------------------------------
  @ImmutableConstructor
  private FixedOvernightCompoundedAnnualRateComputation(double rate, double accrualFactor) {
    this.rate = rate;
    this.accrualFactor = accrualFactor;
    this.simpleRate = (Math.pow(1 + rate, accrualFactor) - 1) / accrualFactor;
  }

  //-------------------------------------------------------------------------
  /**
   * Calculates the simple interest rate associated with the compounded rate.
   * 
   * @return the simple rate
   */
  public double getSimpleRate() {
    return simpleRate;
  }

  //-------------------------------------------------------------------------
  @Override
  public void collectIndices(ImmutableSet.Builder<Index> builder) {
    // no indices to add
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code FixedOvernightCompoundedAnnualRateComputation}.
   * @return the meta-bean, not null
   */
  public static FixedOvernightCompoundedAnnualRateComputation.Meta meta() {
    return FixedOvernightCompoundedAnnualRateComputation.Meta.INSTANCE;
  }

  static {
    MetaBean.register(FixedOvernightCompoundedAnnualRateComputation.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  @Override
  public FixedOvernightCompoundedAnnualRateComputation.Meta metaBean() {
    return FixedOvernightCompoundedAnnualRateComputation.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the fixed rate for overnight compounding.
   * A 5% rate will be expressed as 0.05.
   * @return the value of the property
   */
  public double getRate() {
    return rate;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the accrual factor.
   * @return the value of the property
   */
  public double getAccrualFactor() {
    return accrualFactor;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      FixedOvernightCompoundedAnnualRateComputation other = (FixedOvernightCompoundedAnnualRateComputation) obj;
      return JodaBeanUtils.equal(rate, other.rate) &&
          JodaBeanUtils.equal(accrualFactor, other.accrualFactor);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(rate);
    hash = hash * 31 + JodaBeanUtils.hashCode(accrualFactor);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("FixedOvernightCompoundedAnnualRateComputation{");
    buf.append("rate").append('=').append(JodaBeanUtils.toString(rate)).append(',').append(' ');
    buf.append("accrualFactor").append('=').append(JodaBeanUtils.toString(accrualFactor));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code FixedOvernightCompoundedAnnualRateComputation}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code rate} property.
     */
    private final MetaProperty<Double> rate = DirectMetaProperty.ofImmutable(
        this, "rate", FixedOvernightCompoundedAnnualRateComputation.class, Double.TYPE);
    /**
     * The meta-property for the {@code accrualFactor} property.
     */
    private final MetaProperty<Double> accrualFactor = DirectMetaProperty.ofImmutable(
        this, "accrualFactor", FixedOvernightCompoundedAnnualRateComputation.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "rate",
        "accrualFactor");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3493088:  // rate
          return rate;
        case -1540322338:  // accrualFactor
          return accrualFactor;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends FixedOvernightCompoundedAnnualRateComputation> builder() {
      return new FixedOvernightCompoundedAnnualRateComputation.Builder();
    }

    @Override
    public Class<? extends FixedOvernightCompoundedAnnualRateComputation> beanType() {
      return FixedOvernightCompoundedAnnualRateComputation.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code rate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> rate() {
      return rate;
    }

    /**
     * The meta-property for the {@code accrualFactor} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> accrualFactor() {
      return accrualFactor;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 3493088:  // rate
          return ((FixedOvernightCompoundedAnnualRateComputation) bean).getRate();
        case -1540322338:  // accrualFactor
          return ((FixedOvernightCompoundedAnnualRateComputation) bean).getAccrualFactor();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code FixedOvernightCompoundedAnnualRateComputation}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<FixedOvernightCompoundedAnnualRateComputation> {

    private double rate;
    private double accrualFactor;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3493088:  // rate
          return rate;
        case -1540322338:  // accrualFactor
          return accrualFactor;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 3493088:  // rate
          this.rate = (Double) newValue;
          break;
        case -1540322338:  // accrualFactor
          this.accrualFactor = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public FixedOvernightCompoundedAnnualRateComputation build() {
      return new FixedOvernightCompoundedAnnualRateComputation(
          rate,
          accrualFactor);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("FixedOvernightCompoundedAnnualRateComputation.Builder{");
      buf.append("rate").append('=').append(JodaBeanUtils.toString(rate)).append(',').append(' ');
      buf.append("accrualFactor").append('=').append(JodaBeanUtils.toString(accrualFactor));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
