/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.swaption;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.product.common.SettlementType;

/**
 * Defines the cash settlement type for the payoff of a swaption.
 * <p>
 * The settlement type is {@link SettlementType#CASH}, This means that a cash amount is paid
 * by the short party to the long party at the exercise date.
 */
@BeanDefinition(builderScope = "private")
public final class CashSwaptionSettlement
    implements SwaptionSettlement, ImmutableBean, Serializable {

  /**
   * The settlement date.
   * <p>
   * The payoff of the option is settled at this date.
   */
  @PropertyDefinition(validate = "notNull")
  private final LocalDate settlementDate;

  /**
   * The cash settlement method.
   * <p>
   * The settlement rate of the cash settled swaption is specified by respective cash settlement methods.
   */
  @PropertyDefinition(validate = "notNull")
  private final CashSwaptionSettlementMethod method;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from the settlement date and method.
   * 
   * @param settlementDate  the settlement date
   * @param method  the settlement method
   * @return the settlement
   */
  public static CashSwaptionSettlement of(LocalDate settlementDate, CashSwaptionSettlementMethod method) {
    return new CashSwaptionSettlement(settlementDate, method);
  }

  //-------------------------------------------------------------------------
  @Override
  public SettlementType getSettlementType() {
    return SettlementType.CASH;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CashSwaptionSettlement}.
   * @return the meta-bean, not null
   */
  public static CashSwaptionSettlement.Meta meta() {
    return CashSwaptionSettlement.Meta.INSTANCE;
  }

  static {
    MetaBean.register(CashSwaptionSettlement.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private CashSwaptionSettlement(
      LocalDate settlementDate,
      CashSwaptionSettlementMethod method) {
    JodaBeanUtils.notNull(settlementDate, "settlementDate");
    JodaBeanUtils.notNull(method, "method");
    this.settlementDate = settlementDate;
    this.method = method;
  }

  @Override
  public CashSwaptionSettlement.Meta metaBean() {
    return CashSwaptionSettlement.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the settlement date.
   * <p>
   * The payoff of the option is settled at this date.
   * @return the value of the property, not null
   */
  public LocalDate getSettlementDate() {
    return settlementDate;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the cash settlement method.
   * <p>
   * The settlement rate of the cash settled swaption is specified by respective cash settlement methods.
   * @return the value of the property, not null
   */
  public CashSwaptionSettlementMethod getMethod() {
    return method;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CashSwaptionSettlement other = (CashSwaptionSettlement) obj;
      return JodaBeanUtils.equal(settlementDate, other.settlementDate) &&
          JodaBeanUtils.equal(method, other.method);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(settlementDate);
    hash = hash * 31 + JodaBeanUtils.hashCode(method);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("CashSwaptionSettlement{");
    buf.append("settlementDate").append('=').append(JodaBeanUtils.toString(settlementDate)).append(',').append(' ');
    buf.append("method").append('=').append(JodaBeanUtils.toString(method));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code CashSwaptionSettlement}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code settlementDate} property.
     */
    private final MetaProperty<LocalDate> settlementDate = DirectMetaProperty.ofImmutable(
        this, "settlementDate", CashSwaptionSettlement.class, LocalDate.class);
    /**
     * The meta-property for the {@code method} property.
     */
    private final MetaProperty<CashSwaptionSettlementMethod> method = DirectMetaProperty.ofImmutable(
        this, "method", CashSwaptionSettlement.class, CashSwaptionSettlementMethod.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "settlementDate",
        "method");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -295948169:  // settlementDate
          return settlementDate;
        case -1077554975:  // method
          return method;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends CashSwaptionSettlement> builder() {
      return new CashSwaptionSettlement.Builder();
    }

    @Override
    public Class<? extends CashSwaptionSettlement> beanType() {
      return CashSwaptionSettlement.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code settlementDate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LocalDate> settlementDate() {
      return settlementDate;
    }

    /**
     * The meta-property for the {@code method} property.
     * @return the meta-property, not null
     */
    public MetaProperty<CashSwaptionSettlementMethod> method() {
      return method;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -295948169:  // settlementDate
          return ((CashSwaptionSettlement) bean).getSettlementDate();
        case -1077554975:  // method
          return ((CashSwaptionSettlement) bean).getMethod();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code CashSwaptionSettlement}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<CashSwaptionSettlement> {

    private LocalDate settlementDate;
    private CashSwaptionSettlementMethod method;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -295948169:  // settlementDate
          return settlementDate;
        case -1077554975:  // method
          return method;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -295948169:  // settlementDate
          this.settlementDate = (LocalDate) newValue;
          break;
        case -1077554975:  // method
          this.method = (CashSwaptionSettlementMethod) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public CashSwaptionSettlement build() {
      return new CashSwaptionSettlement(
          settlementDate,
          method);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("CashSwaptionSettlement.Builder{");
      buf.append("settlementDate").append('=').append(JodaBeanUtils.toString(settlementDate)).append(',').append(' ');
      buf.append("method").append('=').append(JodaBeanUtils.toString(method));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
