// Copyright  2014,2015 ~ Optersoft SL

package os.serial.internal;

public class Bytes {

	private static char[] Hex = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

	public static class Reader {

		private int i = 0;
		private final byte[] bytes;

		public Reader(byte[] bytes) {
			this.bytes = bytes;
		}

		public int Int() {

			int value = int_read(bytes, i);
			i += int_size(value);
			return value;

		}

		public String utf8() {
			String result = utf8_read(bytes, i);
			i += utf8_size(bytes, i);
			return result;
		}

	}

	public static class Buffer {

		public byte[] array = new byte[256];
		public int index = 0;

		public void add(byte b) {
			ensureSize(index + 1);
			array[index++] = b;
		}

		public void put(byte b) {
			array[index++] = b;
		}

		public byte[] bytes() {

			byte[] result = new byte[index];
			System.arraycopy(array, 0, result, 0, index);
			return result;
		}

		private void ensureSize(int n) {
			if (n > array.length)
				increaseSize(n);
		}

		public void needs(int needs) {
			int n = index + needs;
			if (n > array.length)
				increaseSize(n);
		}

		private void increaseSize(int n) {
			int newSize = array.length * 2;
			while (n > newSize)
				newSize = newSize * 2;

			byte[] newArray = new byte[newSize];
			System.arraycopy(array, 0, newArray, 0, index);
			array = newArray;

		}
	}

	public static void boolean_write(Buffer writer, boolean b) {
		if (b)
			writer.add((byte) 1);
		else
			writer.add((byte) 0);

	}

	public static boolean boolean_read(byte[] bytes, int i) {
		return bytes[i] != 0;
	}

	public static double double_read(byte[] bytes, int i) {
		return Double.longBitsToDouble(long_read(bytes, i));
	}

	public static void double_write(Buffer writer, double d) {
		long_write(writer, Double.doubleToLongBits(d));
	}

	public static double float_read(byte[] bytes, int i) {
		return Float.intBitsToFloat(int_read(bytes, i));
	}

	public static int int_read(byte[] bytes, int index) {

		int result = 0;
		for (int shift = 0; shift < 32; shift += 7) {
			final int b = bytes[index++] & 0xff;
			result |= (b & 0x7F) << shift;
			if ((b & 0x80) == 0)
				return result;
		}

		throw new Error("Malformed var int");
	}

	public static int int_size(byte[] bytes, int index) {

		int b1 = bytes[index];
		if (b1 >= 0)
			return 1;
		else {
			int b2 = bytes[++index];
			if (b2 >= 0)
				return 2;
			else {
				index++;
				int skip = 2;
				int until = index + 3;
				for (; index < until; index++) {
					skip++;
					int b = bytes[index];
					if (b >= 0)
						return skip;
				}
				throw new Error("Malformed var int");

			}
		}
	}

	public static int int_size(int value) {
		if ((value & (0xffffffff << 7)) == 0)
			return 1;
		if ((value & (0xffffffff << 14)) == 0)
			return 2;
		if ((value & (0xffffffff << 21)) == 0)
			return 3;
		if ((value & (0xffffffff << 28)) == 0)
			return 4;
		return 5;
	}

	public static void int_write(byte[] bytes, int index, int i) {

		while (true) {
			if ((i & ~0x7F) == 0) {
				bytes[index++] = (byte) i;
				return;
			} else {
				bytes[index++] = (byte) ((i & 0x7F) | 0x80);
				i >>>= 7;
			}
		}
	}

	public static void wInt(Buffer buffer, int i) {

		while (true) {
			if ((i & ~0x7F) == 0) {
				buffer.add((byte) i);
				return;
			} else {
				buffer.add((byte) ((i & 0x7F) | 0x80));
				i >>>= 7;
			}
		}
	}

	public static long long_read(byte[] bytes, int i) {

		return ((long) bytes[i++]) << 56 | ((long) bytes[i++] & 0xff) << 48 | ((long) bytes[i++] & 0xff) << 40
				| ((long) bytes[i++] & 0xff) << 32 | ((long) bytes[i++] & 0xff) << 24 | (bytes[i++] & 0xff) << 16
				| (bytes[i++] & 0xff) << 8 | bytes[i++] & 0xff;

	}

	public static void long_write(Buffer buffer, long value) {

		buffer.needs(8);
		buffer.put((byte) (value >> 56));
		buffer.put((byte) (value >> 48));
		buffer.put((byte) (value >> 40));
		buffer.put((byte) (value >> 32));
		buffer.put((byte) (value >> 24));
		buffer.put((byte) (value >> 16));
		buffer.put((byte) (value >> 8));
		buffer.put((byte) (value));

	}

	public static long longVar_read(byte[] bytes, int i) {

		long result = 0;
		int shift = 0;

		while (shift < 64) {
			int value = bytes[i++] & 0xff;
			result |= ((long) (value & 0x7F)) << shift;
			if ((value & 0x80) == 0)
				return result;
			shift += 7;
		}

		throw new Error("Malformed var long");

	}

	public static int longVar_size(long value) {
		if ((value & (0xffffffffffffffffL << 7)) == 0)
			return 1;
		if ((value & (0xffffffffffffffffL << 14)) == 0)
			return 2;
		if ((value & (0xffffffffffffffffL << 21)) == 0)
			return 3;
		if ((value & (0xffffffffffffffffL << 28)) == 0)
			return 4;
		if ((value & (0xffffffffffffffffL << 35)) == 0)
			return 5;
		if ((value & (0xffffffffffffffffL << 42)) == 0)
			return 6;
		if ((value & (0xffffffffffffffffL << 49)) == 0)
			return 7;
		if ((value & (0xffffffffffffffffL << 56)) == 0)
			return 8;
		if ((value & (0xffffffffffffffffL << 63)) == 0)
			return 9;
		return 10;
	}

	public static int longVar_skip(byte[] bytes, int index) {

		int shift = 0;

		while (shift < 64) {
			int value = bytes[index++] & 0xff;
			if ((value & 0x80) == 0)
				return index;
			shift += 7;
		}

		throw new Error("Malformed var long");

	}

	public static void longVar_write(Buffer buffer, long l) {

		while (true) {
			if ((l & ~0x7F) == 0) {
				buffer.add((byte) l);
				return;
			} else {
				buffer.add((byte) ((l & 0x7F) | 0x80));
				l >>>= 7;
			}
		}
	}

	public static int longVar_write(byte[] bytes, int index, long l) {

		while (true) {
			if ((l & ~0x7F) == 0) {
				bytes[index++] = (byte) l;
				return index;
			} else {
				bytes[index++] = (byte) ((l & 0x7F) | 0x80);
				l >>>= 7;
			}
		}

	}

	public static String utf8_read(byte[] bytes, int index) {

		int length = int_read(bytes, index);
		int bi = index + int_size(bytes, index);
		length = bi + length;

		int offset = bi;
		int charCount = 0;
		while (offset < length) {
			switch ((bytes[offset] & 0xff) >> 4) {
			case 0:
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
				offset++;
				break;
			case 12:
			case 13:
				offset += 2;
				break;
			case 14:
				offset += 3;
				break;
			default:
				throw new IllegalArgumentException();
			}
			charCount++;
		}

		char[] chars = new char[charCount];

		int ci = 0;
		int char1, char2, char3;
		while (bi < length) {

			char1 = bytes[bi++] & 0xff;
			byte test = (byte) ((char1 & 0xff) >> 4);

			if (test < 8) {
				chars[ci++] = (char) char1;
			} else if (test == 12 || test == 13) {
				char2 = bytes[bi++];
				if ((char2 & 0xC0) != 0x80) {
					throw new IllegalArgumentException();
				}
				chars[ci++] = (char) (((char1 & 0x1F) << 6) | (char2 & 0x3F));
			} else if (test == 14) {
				char2 = bytes[bi++];
				char3 = bytes[bi++];
				if (((char2 & 0xC0) != 0x80) || ((char3 & 0xC0) != 0x80))
					throw new IllegalArgumentException();
				chars[ci++] = (char) (((char1 & 0x0F) << 12) | ((char2 & 0x3F) << 6) | ((char3 & 0x3F) << 0));
			} else
				throw new IllegalArgumentException();
		}

		return new String(chars);
	}

	public static int utf8_size(byte[] bytes, int index) {

		int size = int_read(bytes, index);
		return int_size(size) + size;
	}

	public static void utf8_write(Buffer buffer, String str) {

		int len = 0;
		for (int i = 0; i < str.length(); i++) {
			char c = str.charAt(i);
			if ((c >= 0x0001) && (c <= 0x007F))
				len++;
			else if (c > 0x07FF)
				len += 3;
			else
				len += 2;
		}

		wInt(buffer, len);

		for (int i = 0; i < str.length(); i++) {
			char c = str.charAt(i);

			if ((c >= 0x0001) && (c <= 0x007F)) {
				buffer.add((byte) c);
			} else if (c > 0x07FF) {
				buffer.add((byte) (0xE0 | ((c >> 12) & 0x0F)));
				buffer.add((byte) (0x80 | ((c >> 6) & 0x3F)));
				buffer.add((byte) (0x80 | ((c >> 0) & 0x3F)));
			} else {
				buffer.add((byte) (0xC0 | ((c >> 6) & 0x1F)));
				buffer.add((byte) (0x80 | ((c >> 0) & 0x3F)));
			}
		}

	}

}
