// Copyright 2014,2015 ~ Optersoft SL

package os.serial.internal

import org.objenesis.ObjenesisStd
import scala.collection.mutable.HashMap
import org.objenesis.instantiator.ObjectInstantiator
import os.serial.internal.Bytes._
import scala.concurrent._
import os.serial.internal._
import os.serial.SerialError
import os.serial.Serial

object Reify extends Types {

  private val objenesis = new ObjenesisStd()
  private val map = new HashMap[Int, ObjectInstantiator[_]]

  private val bytesField = {
    val field = classOf[Serial].getDeclaredField("$bytes")
    field.setAccessible(true)
    field
  }

  private val indexField = {
    val field = classOf[Serial].getDeclaredField("$index")
    field.setAccessible(true)
    field
  }

  def apply[S <: Serial](bytes: Array[Byte], index: Int = 0): S = {

    var i = index
    if (index == 0) i += 1

    val id = int_read(bytes, i)

    val instantiator = map.getOrElseUpdate(id,
      Metadata(id) match {
        case None => throw new SerialError(s"Class id not found: ${id.toHexString}")
        case Some(clazz) => objenesis.getInstantiatorOf(clazz)
      })

    val instance = instantiator.newInstance().asInstanceOf[S]
    bytesField.set(instance, bytes)
    indexField.set(instance, index)

    instance

  }

  def field[T](bytes: Array[Byte], id: Int, index: Int = 0): T = {

    val tpeIndex = findField(bytes, id, index); val i = tpeIndex.asInstanceOf[Int]

    if (i == -1) null.asInstanceOf[T]
    else {

      val result = (tpeIndex >> 56) match {
        case INTEGER => readInteger(bytes, i)
        case STRING => readString(bytes, i)
        case SERIAL => apply(bytes, skipInt(bytes, i))
        case t =>
          //Debug.printBytes(bytes, index)
          throw new Exception("Unknow type " + t)

      }

      result.asInstanceOf[T]
    }

  }

  def fieldInt(bytes: Array[Byte], id: Int, index: Int): Int = {

    findField(bytes, id, index)

    0
  }

  def findField(bytes: Array[Byte], id: Int, index: Int): Long = {

    var i = index
    if (i == 0) i = 1 // serial version 
    i = skipInt(bytes, i) // class id 

    // ids

    var id1 = id & 0xF; var id2 = 0; var id3 = 0; var id4 = 0; var id5 = 0
    if ((id & 0xFFFFFFF0) != 0) {
      id1 = id1 | 0x10; id2 = (id >>> 4) & 0x7f
      if ((id & 0xFFFFF800) != 0) {
        id2 = id2 | 0xFFFFFF80; id3 = (id >>> 11) & 0x7f
        if ((id & 0xFFFC0000) != 0) {
          id3 = id3 | 0xFFFFFF80; id4 = (id >>> 18) & 0x7f
          if ((id & 0xFE000000) != 0) {
            id4 = id4 | 0xFFFFFF80; id5 = (id >>> 25) & 0x7f
          }
        }
      }
    }

    // find

    var byte: Int = bytes(i); i += 1;
    while (byte != -1) {

      //Debug.printTypeId(bytes, i - 1)

      val tpe = byte & 0xE0L; byte = byte & 0x1F

      if (byte != id1) {
        if (byte > 0xF)
          i = skipInteger(bytes, i)
      } else {
        if (byte <= 0xF) return (tpe << 56) | i else {
          byte = bytes(i); i += 1; if (byte == id2) {
            if (id3 == 0) return (tpe << 56) | i else {
              byte = bytes(i); i += 1; if (byte == id3) {
                if (id4 == 0) return (tpe << 56) | i else {
                  byte = bytes(i); i += 1; if (byte == id4) {
                    if (id5 == 0) return (tpe << 56) | i else {
                      byte = bytes(i); if (byte == id5) return (tpe << 56) | (i + 1)
                    }
                  }
                }
              }
            }
          }

          i = skipInteger(bytes, i)
        }
      }

      tpe match {
        case STRING =>
          var length = readInt(bytes, i)
          i += int_size(bytes, i)
          i += length
        case INTEGER =>
          i = skipInteger(bytes, i)
        case _ => throw new RuntimeException(s"Unkown type $tpe ($i)")
      }

      byte = bytes(i); i += 1
    }

    -1
  }

  def readInt(bytes: Array[Byte], index: Int): Int = {

    var result = 0; var shift = 0; var i = index
    while (shift < 32) {
      val b = bytes(i) & 0xff; i += 1
      result |= (b & 0x7F) << shift
      if ((b & 0x80) == 0)
        return result

      shift += 7
    }

    throw new Error("Malformed var int");
  }

  def readInteger(bytes: Array[Byte], index: Int) = {
    // TODO
    readInt(bytes, index)
  }

  def readString(bytes: Array[Byte], index: Int) = {

    var bEnd = readInt(bytes, index)
    var bi = index + int_size(bytes, index)
    bEnd = bi + bEnd;

    var offset = bi; var charCount = 0
    while (offset < bEnd) {
      val x = (bytes(offset) & 0xff) >> 4
      if (x <= 7) offset += 1
      else if (x == 12 || x == 13) offset += 2
      else if (x == 14) offset += 3
      else throw new IllegalArgumentException()

      charCount += 1
    }

    val chars = new Array[Char](charCount)

    var ci = 0
    while (bi < bEnd) {

      val char1 = bytes(bi) & 0xff; bi += 1
      val test = ((char1 & 0xff) >> 4).asInstanceOf[Byte]

      if (test < 8) {
        chars(ci) = char1.asInstanceOf[Char]; ci += 1
      } else if (test == 12 || test == 13) {
        val char2 = bytes(bi); bi += 1
        if ((char2 & 0xC0) != 0x80) {
          throw new IllegalArgumentException();
        }
        chars(ci) = (((char1 & 0x1F) << 6) | (char2 & 0x3F)).asInstanceOf[Char]; ci += 1
      } else if (test == 14) {
        val char2 = bytes(bi); bi += 1
        val char3 = bytes(bi); bi += 1
        if (((char2 & 0xC0) != 0x80) || ((char3 & 0xC0) != 0x80))
          throw new IllegalArgumentException();
        chars(ci) = (((char1 & 0x0F) << 12) | ((char2 & 0x3F) << 6) |
          ((char3 & 0x3F) << 0)).asInstanceOf[Char]; ci += 1
      } else
        throw new IllegalArgumentException()
    }

    new String(chars)

  }

  def skipInt(bytes: Array[Byte], index: Int): Int = {

    var i = index; var byte = bytes(i); i += 1

    if (byte < 0) {
      byte = bytes(i); i += 1
      if (byte < 0) {
        do {
          byte = bytes(i); i += 1
        } while (byte < 0)
        // TODO throw new Error("Malformed var int") if more than 5 bytes;
      }
    }

    i
  }

  def skipInteger(bytes: Array[Byte], index: Int): Int = {

    var i = index; var byte = bytes(i); i += 1

    while (byte < 0) {
      byte = bytes(i); i += 1
    }

    i

  }

}