// Copyright 2014,2015 ~ Optersoft SL

package os.serial.internal

import os.serial.Serial
import os.serial.internal.Bytes._
import os.serial.SerialError

object Serialize extends Types {

  implicit def serialToBytes(serial: Serial): Array[Byte] = apply(serial)

  def apply(serial: Serial): Array[Byte] = {

    val buffer = new ByteBuffer()
    buffer.add(0)
    writeSerial(buffer, serial)
    buffer.bytes()
  }

  def data(buffer: ByteBuffer) = {
    val index = buffer.index
    buffer.add(0)
    index
  }

  def data(buffer: ByteBuffer, start: Int, updateSize: Boolean) {

    val size = buffer.index - start

    if (size > 127) {

      if (size > 0x3FFF) {
        throw new Exception("Not implemented yet!")
      }

      val array = buffer.array
      if (array(start) >= 0) {

        buffer.add(0) // increase 1
        var i = buffer.index - 1
        while (i > start) {
          array(i + 1) = array(i); i -= 1

        }

        array(start) = -1
        array(start + 1) = 0
      }
    }

    if (updateSize) {
      val index = buffer.index
      buffer.index = start
      writeInt(buffer, index - start)
      buffer.index = index
    }

  }

  def writeInt(buffer: ByteBuffer, int: Int) {

    var i = int

    while (true) {
      if ((i & ~0x7F) == 0) {
        buffer.add(i.asInstanceOf[Byte]);
        return
      } else {
        buffer.add(((i & 0x7F) | 0x80).asInstanceOf[Byte])
        i >>>= 7
      }
    }

  }

  def writeInteger(buffer: ByteBuffer, int: Int) {
    // TODO
    writeInt(buffer, int)

  }

  def writeSeq(buffer: ByteBuffer, seq: Seq[_]) {

    val start = data(buffer)
    writeInt(buffer, seq.size)

    var i = 0; while (i < seq.size) {
      val elem = seq(i)
      // write eleme
      data(buffer, start, false)
      i += 1
    }

    data(buffer, start, true)

  }

  def writeSerial(buffer: ByteBuffer, serial: Serial) {

    val start = buffer.index
    val embedded = start > 1
    if (embedded)
      data(buffer)

    val mirror = Reflect(serial.getClass())

    writeInt(buffer, mirror.id)

    mirror.attributes foreach { attribute =>

      val value = attribute.get(serial)
      if (value != null) {
        value match {
          case int: Int =>
            if (int != 0) {
              writeTypeId(buffer, INTEGER, attribute.id)
              writeInteger(buffer, int)
            }
          case str: String =>
            if (str.length() != 0) {
              writeTypeId(buffer, STRING, attribute.id)
              writeString(buffer, str)
            }
          case serial: Serial =>
            writeTypeId(buffer, SERIAL, attribute.id)
            writeSerial(buffer, serial)

          case seq: Seq[_] =>
            writeTypeId(buffer, SEQ, attribute.id)
            writeSeq(buffer, seq)

          case other => //println(other)
        }
      }

      if (embedded)
        data(buffer, start, false)

    }

    if (embedded)
      data(buffer, start, true)

    buffer.add(0xff.asInstanceOf[Byte])

  }

  def writeString(buffer: ByteBuffer, string: String) {

    var len = 0; var i = 0; val strLength = string.length
    while (i < strLength) {
      val char = string.charAt(i)
      if ((char >= 0x0001) && (char <= 0x007F))
        len += 1;
      else if (char > 0x07FF)
        len += 3;
      else
        len += 2;

      i += 1
    }

    writeInt(buffer, len)

    i = 0
    while (i < strLength) {
      val char = string.charAt(i)

      if ((char >= 0x0001) && (char <= 0x007F)) {
        buffer.add(char.asInstanceOf[Byte])
      } else if (char > 0x07FF) {
        buffer.add((0xE0 | ((char >> 12) & 0x0F)).asInstanceOf[Byte])
        buffer.add((0x80 | ((char >> 6) & 0x3F)).asInstanceOf[Byte])
        buffer.add((0x80 | ((char >> 0) & 0x3F)).asInstanceOf[Byte])
      } else {
        buffer.add((0xC0 | ((char >> 6) & 0x1F)).asInstanceOf[Byte])
        buffer.add((0x80 | ((char >> 0) & 0x3F)).asInstanceOf[Byte])
      }

      i += 1

    }

  }

  def writeTypeId(buffer: ByteBuffer, tpe: Byte, id: Int) {

    var id1 = id & 0xF
    if (id <= 0xF && id >= 0) {

      val byte = (tpe | id1).asInstanceOf[Byte]
      buffer add byte

    } else {
      val byte = (tpe | id1 | 0x10).asInstanceOf[Byte]
      buffer add byte

      var i = id >>> 4
      while (true) {
        if ((i & ~0x7F) == 0) {
          buffer.add(i.asInstanceOf[Byte]);
          return
        } else {
          buffer.add(((i & 0x7F) | 0x80).asInstanceOf[Byte])
          i >>>= 7
        }
      }
    }
  }

}