/*-
 * Copyright (C) 2011, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This file was distributed by Oracle as part of a version of Oracle NoSQL
 * Database made available at:
 *
 * http://www.oracle.com/technetwork/database/database-technologies/nosqldb/downloads/index.html
 *
 * Please see the LICENSE file included in the top-level directory of the
 * appropriate version of Oracle NoSQL Database for a copy of the license and
 * additional information.
 */

package oracle.kv.impl.query.runtime;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;

import oracle.kv.impl.api.table.FieldDefImpl;
import oracle.kv.impl.api.table.FieldValueImpl;
import oracle.kv.impl.api.table.RecordValueImpl;
import oracle.kv.impl.api.table.TupleValue;
import oracle.kv.impl.async.IterationHandleNotifier;
import oracle.kv.impl.query.compiler.Expr;
import oracle.kv.impl.query.compiler.QueryFormatter;
import oracle.kv.impl.query.compiler.SortSpec;
import oracle.kv.impl.query.runtime.PlanIterState.StateEnum;
import oracle.kv.impl.util.SizeOf;

public class SortIter extends PlanIter {

    private class CompareFunction implements Comparator<FieldValueImpl> {

        @Override
        public int compare(FieldValueImpl v1, FieldValueImpl v2) {

            if (theInputType.isRecord()) {

                return ReceiveIter.compareRecords(
                    (RecordValueImpl)v1,
                    (RecordValueImpl)v2,
                    theSortFieldPositions,
                    theSortSpecs);
            }

            return ReceiveIter.compareAtomics(v1, v2, 0, theSortSpecs);
        }
    }

    private static class SortIterState extends PlanIterState {

        final ArrayList<FieldValueImpl> theResults;

        int theCurrResult;

        CompareFunction theComparator;

        long theMemoryConsumption = theFixedMemoryConsumption;

        public SortIterState(SortIter iter) {
            super();
            theResults = new ArrayList<FieldValueImpl>(4096);
            theComparator = iter.new CompareFunction();
        }

        @Override
        public void done() {
            super.done();
            theCurrResult = 0;
            theResults.clear();
        }

        @Override
        public void reset(PlanIter iter) {
            super.reset(iter);
            theCurrResult = 0;
            theResults.clear();
            theMemoryConsumption = theFixedMemoryConsumption;
        }

        @Override
        public void close() {
            super.close();
            theResults.clear();
        }
    }

    private static final long theFixedMemoryConsumption =
        (2 * SizeOf.OBJECT_REF_OVERHEAD +
         SizeOf.ARRAYLIST_OVERHEAD +
         SizeOf.OBJECT_OVERHEAD +
         12);

    private final PlanIter theInput;

    private final FieldDefImpl theInputType;

    private final int[] theSortFieldPositions;

    private final SortSpec[] theSortSpecs;

    public SortIter(
        Expr e,
        int resultReg,
        PlanIter input,
        FieldDefImpl inputType,
        int[] sortFieldPositions,
        SortSpec[] sortSpecs) {

        super(e, resultReg);

        theInput = input;
        theInputType = inputType;
        theSortFieldPositions = sortFieldPositions;
        theSortSpecs = sortSpecs;
    }

    public SortIter(DataInput in, short serialVersion) throws IOException {

        super(in, serialVersion);
        theInput = deserializeIter(in, serialVersion);
        theInputType = (FieldDefImpl) deserializeFieldDef(in, serialVersion);
        theSortFieldPositions = deserializeIntArray(in, serialVersion);
        theSortSpecs = deserializeSortSpecs(in, serialVersion);
    }

    @Override
    public void writeFastExternal(DataOutput out, short serialVersion)
            throws IOException {

        super.writeFastExternal(out, serialVersion);
        serializeIter(theInput, out, serialVersion);
        serializeFieldDef(theInputType, out, serialVersion);
        serializeIntArray(theSortFieldPositions, out, serialVersion);
        serializeSortSpecs(theSortSpecs, out, serialVersion);
    }

    @Override
    public PlanIterKind getKind() {
        return PlanIterKind.SORT;
    }

    @Override
    public void setIterationHandleNotifier(
        IterationHandleNotifier iterHandleNotifier) {
        theInput.setIterationHandleNotifier(iterHandleNotifier);
    }

    @Override
    public void open(RuntimeControlBlock rcb) {
        SortIterState state = new SortIterState(this);
        rcb.setState(theStatePos, state);
        rcb.incMemoryConsumption(state.theMemoryConsumption);
        theInput.open(rcb);
    }

    @Override
    public void reset(RuntimeControlBlock rcb) {

        theInput.reset(rcb);
        SortIterState state = (SortIterState)rcb.getState(theStatePos);
        rcb.decMemoryConsumption(state.theMemoryConsumption -
                                 theFixedMemoryConsumption);
        state.reset(this);
    }

    @Override
    public void close(RuntimeControlBlock rcb) {

        PlanIterState state = rcb.getState(theStatePos);
        if (state == null) {
            return;
        }

        theInput.close(rcb);
        state.close();
    }

    @Override
    public boolean next(RuntimeControlBlock rcb) {
        return nextInternal(rcb);
    }

    @Override
    public boolean nextLocal(RuntimeControlBlock rcb) {
        return nextInternal(rcb);
    }

    private boolean nextInternal(RuntimeControlBlock rcb) {

        SortIterState state = (SortIterState)rcb.getState(theStatePos);

        if (state.isDone()) {
            return false;
        }

        if (state.isOpen()) {

            boolean more = theInput.next(rcb);

            while (more) {
                FieldValueImpl v = rcb.getRegVal(theInput.getResultReg());

                if (v.isTuple()) {
                    v = ((TupleValue)v).toRecord();
                }

                state.theResults.add(v);
                long sz = v.sizeof() + SizeOf.OBJECT_REF_OVERHEAD;
                state.theMemoryConsumption += sz;
                rcb.incMemoryConsumption(sz);

                more = theInput.next(rcb);
            }

            state.theResults.sort(state.theComparator);

            state.setState(StateEnum.RUNNING);
        }

        if (state.theCurrResult < state.theResults.size()) {

            rcb.setRegVal(theResultReg,
                          state.theResults.get(state.theCurrResult));
            ++state.theCurrResult;
            return true;
        }

        state.done();
        return false;
    }

    @Override
    protected void displayContent(StringBuilder sb, QueryFormatter formatter) {
        theInput.display(sb, formatter);
    }
}
