/*-
 * Copyright (C) 2011, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This file was distributed by Oracle as part of a version of Oracle NoSQL
 * Database made available at:
 *
 * http://www.oracle.com/technetwork/database/database-technologies/nosqldb/downloads/index.html
 *
 * Please see the LICENSE file included in the top-level directory of the
 * appropriate version of Oracle NoSQL Database for a copy of the license and
 * additional information.
 */

package oracle.kv.impl.rep.admin;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;

import com.sleepycat.je.rep.InsufficientLogException;

import oracle.kv.KVSecurityException;
import oracle.kv.impl.fault.ClientAccessException;
import oracle.kv.impl.fault.ProcessExitCode;
import oracle.kv.impl.fault.ServiceFaultHandler;
import oracle.kv.impl.rep.RepNodeService;

/**
 * Specializes the RepNodeServiceFaultHandler so that all thrown exceptions are
 * wrapped inside a RepNodeAdminFaultException.
 */
public class RepNodeAdminFaultHandler extends ServiceFaultHandler {

    /**
     * Tracks the number of requests that are currently active.
     */
    private final AtomicInteger activeRequests = new AtomicInteger(0);

    public RepNodeAdminFaultHandler(RepNodeService repNodeService,
                                    Logger logger,
                                    ProcessExitCode defaultExitCode) {
        super(repNodeService, logger, defaultExitCode);
    }

    public int getActiveRequests() {
        return activeRequests.get();
    }

    /**
     * Wrap it inside a RepNodeAdminFaultException.
     */
    @Override
    protected RuntimeException getThrowException(RuntimeException fault) {

        if (fault instanceof ClientAccessException) {
            /*
             * This is a security exception generated by the client.
             * Unwrap it so that the client sees it in its orginal form.
             */
            return ((ClientAccessException) fault).getCause();
        }

        /*
         * Security exceptions are passed through.
         */
        if (fault instanceof KVSecurityException) {
            return fault;
        }

        return new RepNodeAdminFaultException(fault);
    }

    @Override
    public ProcessExitCode getExitCode(RuntimeException requestException,
                                          ProcessExitCode exitCode) {
        /*
         * Note that we don't have to take any corrective action, that is
         * "handle" the ILE exception, the state change listener should have
         * noticed the transition to detached state and started a network
         * restore.
         */
        return (requestException instanceof InsufficientLogException) ?
            null : super.getExitCode(requestException, exitCode);
    }


    @Override
    public <R, E extends Exception> R execute(Operation<R, E> operation)
        throws E {

        activeRequests.incrementAndGet();
        try {
            return super.execute(operation);
        } finally {
            activeRequests.decrementAndGet();
        }
    }

    @Override
    public <R> R execute(SimpleOperation<R> operation) {

        activeRequests.incrementAndGet();
        try {
            return super.execute(operation);
        } finally {
            activeRequests.decrementAndGet();
        }
    }


    @Override
    public <E extends Exception> void execute(Procedure<E> proc)
        throws E {

        activeRequests.incrementAndGet();
        try {
            super.execute(proc);
        } finally {
            activeRequests.decrementAndGet();
        }
    }


    @Override
    public void execute(SimpleProcedure proc) {

        activeRequests.incrementAndGet();
        try {
            super.execute(proc);
        } finally {
            activeRequests.decrementAndGet();
        }
    }
}
