/*-
 *
 *  This file is part of Oracle NoSQL Database
 *  Copyright (C) 2011, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 * If you have received this file as part of Oracle NoSQL Database the
 * following applies to the work as a whole:
 *
 *   Oracle NoSQL Database server software is free software: you can
 *   redistribute it and/or modify it under the terms of the GNU Affero
 *   General Public License as published by the Free Software Foundation,
 *   version 3.
 *
 *   Oracle NoSQL Database is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Affero General Public License for more details.
 *
 * If you have received this file as part of Oracle NoSQL Database Client or
 * distributed separately the following applies:
 *
 *   Oracle NoSQL Database client software is free software: you can
 *   redistribute it and/or modify it under the terms of the Apache License
 *   as published by the Apache Software Foundation, version 2.0.
 *
 * You should have received a copy of the GNU Affero General Public License
 * and/or the Apache License in the LICENSE file along with Oracle NoSQL
 * Database client or server distribution.  If not, see
 * <http://www.gnu.org/licenses/>
 * or
 * <http://www.apache.org/licenses/LICENSE-2.0>.
 *
 * An active Oracle commercial licensing agreement for this product supersedes
 * these licenses and in such case the license notices, but not the copyright
 * notice, may be removed by you in connection with your distribution that is
 * in accordance with the commercial licensing terms.
 *
 * For more information please contact:
 *
 * berkeleydb-info_us@oracle.com
 *
 */

package oracle.kv.impl.api.table;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import oracle.kv.impl.admin.IllegalCommandException;
import oracle.kv.table.FieldDef;
import oracle.kv.table.FieldRange;
import oracle.kv.table.FieldValue;
import oracle.kv.table.Index;
import oracle.kv.table.IndexKey;
import oracle.kv.table.RecordValue;
import oracle.kv.table.Table;

import org.codehaus.jackson.node.ArrayNode;
import org.codehaus.jackson.node.ObjectNode;

import com.sleepycat.bind.tuple.TupleInput;
import com.sleepycat.bind.tuple.TupleOutput;

/**
 * Implementation of the Index interface.
 */
public class IndexImpl implements Index, Serializable {

    private static final long serialVersionUID = 1L;
    private final String name;
    private final String description;
    private final TableImpl table;
    private final List<String> fields;
    private IndexStatus status;
    private transient List<IndexField> indexFields;

    public enum IndexStatus {
        /** Index is transient */
        TRANSIENT() {
            @Override
            public boolean isTransient() {
                return true;
            }
        },

        /** Index is being populated */
        POPULATING() {
            @Override
            public boolean isPopulating() {
                return true;
            }
        },

        /** Index is populated and ready for use */
        READY() {
            @Override
            public boolean isReady() {
                return true;
            }
        };

        /**
         * Returns true if this is the {@link #TRANSIENT} type.
         * @return true if this is the {@link #TRANSIENT} type
         */
        public boolean isTransient() {
            return false;
        }

        /**
         * Returns true if this is the {@link #POPULATING} type.
         * @return true if this is the {@link #POPULATING} type
         */
	public boolean isPopulating() {
            return false;
        }

        /**
         * Returns true if this is the {@link #READY} type.
         * @return true if this is the {@link #READY} type
         */
	public boolean isReady() {
            return false;
        }
    }

    public IndexImpl(String name, TableImpl table, List<String> fields,
                     String description) {
        this.name = name;
        this.table = table;
        this.fields = fields;
        this.description = description;
        status = IndexStatus.TRANSIENT;
        validate();
    }

    @Override
    public Table getTable() {
        return table;
    }

    @Override
    public String getName()  {
        return name;
    }

    @Override
    public List<String> getFields() {
        return Collections.unmodifiableList(fields);
    }

    @Override
    public String getDescription()  {
        return description;
    }

    @Override
    public IndexKeyImpl createIndexKey() {
        return new IndexKeyImpl(this);
    }

    @Override
    public IndexKeyImpl createIndexKey(RecordValue value) {
        IndexKeyImpl key = new IndexKeyImpl(this);
        TableImpl.populateRecord(key, value);
        return key;
    }

    @Override
    public IndexKey createIndexKeyFromJson(String jsonInput, boolean exact) {
        return createIndexKeyFromJson
            (new ByteArrayInputStream(jsonInput.getBytes()), exact);
    }

    @Override
    public IndexKey createIndexKeyFromJson(InputStream jsonInput,
                                           boolean exact) {
        IndexKeyImpl key = createIndexKey();
        table.createFromJson(key, jsonInput, exact);
        return key;
    }

    @Override
    public FieldRange createFieldRange(String fieldName) {
        IndexField field = new IndexField(fieldName);
        FieldDef def = findIndexField(field);
        if (def == null) {
            throw new IllegalArgumentException
                ("Field does not exist in table definition: " + fieldName);
        }
        if (!containsField(field)) {
            throw new IllegalArgumentException
                ("Field does not exist in index: " + fieldName);
        }
        return new FieldRange(fieldName, def);
    }

    int numFields() {
        return fields.size();
    }

    /**
     * Returns true if the index comprises only fields from the table's primary
     * key.  Nested types can't be key components so there is no need to handle
     * a complex path.
     */
    public boolean isKeyOnly() {
        for (String field : fields) {
            if (!table.isKeyComponent(field)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Return true if this index has multiple keys/record.  This happens if
     * there is an array in the index.  An index can only contain one array.
     */
    public boolean isMultiKey() {
        for (IndexField field : getIndexFields()) {
            if (field.hasArray()) {
                return true;
            }
        }
        return false;
    }

    public IndexStatus getStatus() {
        return status;
    }

    public void setStatus(IndexStatus status) {
        this.status = status;
    }

    public TableImpl getTableImpl() {
        return table;
    }

    List<String> getFieldsInternal() {
        return fields;
    }

    /**
     * Returns the list of IndexField objects defining the index.  It is
     * transient, and if not yet initialized, initialize it.
     */
    List<IndexField> getIndexFields() {
        if (indexFields == null) {
            initIndexFields();
        }
        return indexFields;
    }

    /**
     * Initializes the transient list of index fields.  Most, if not all
     * callers won't require the synchronization but it doesn't hurt, as it's
     * only done once in the lifetime of this instance.
     */
    private synchronized void initIndexFields() {
        if (indexFields == null) {
            List<IndexField> list = new ArrayList<IndexField>(fields.size());
            for (String field : fields) {
                IndexField indexField = new IndexField(field);

                /* this sets the hasArray state of the IndexField */
                containsArray(indexField);
                list.add(indexField);
            }
            indexFields = list;
        }
    }

    /**
     * If there's an array in the index return its name.
     */
    private String findArray() {
        for (IndexField field : getIndexFields()) {
            if (field.hasArray()) {
                return field.getArrayPath();
            }
        }

        /* this should never happen */
        return null;
    }

    /**
     * Returns true if the (complex) fieldName contains a reference to an
     * array field anywhere in its path, false otherwise.  This also has
     * the effect of setting the array state in the IndexField so that
     * the lookup need not be done twice.
     */
    private boolean containsArray(IndexField field) {
        StringBuilder sb = new StringBuilder();
        Iterator<String> iter = field.iterator();

        FieldDefImpl def = field.getFirstDef();
        if (def.isArray()) {
            field.setArrayPath(iter.next());
            return true;
        }

        sb.append(iter.next()).append(TableImpl.SEPARATOR);
        while (iter.hasNext()) {
            String current = iter.next();
            sb.append(current);
            def = def.findField(current);
            assert def != null;
            if (def.isArray()) {
                field.setArrayPath(sb.toString());
                return true;
            }
            sb.append(TableImpl.SEPARATOR);
        }
        return false;
    }

    /**
     * Extracts an index key from the key and data for this
     * index.  The key has already matched this index.
     *
     * @param key the key bytes
     *
     * @param data the row's data bytes
     *
     * @param keyOnly true if the index only uses key fields.  This
     * optimizes deserialization.
     *
     * @return the byte[] serialization of an index key or null if there
     * is no entry associated with the row, or the row does not match a
     * table record.
     *
     * While not likely it is possible that the record is not actually  a
     * table record and the key pattern happens to match.  Such records
     * will fail to be deserialized and throw an exception.  Rather than
     * treating this as an error, silently ignore it.
     *
     * TODO: maybe make this faster.  Right now it turns the key and data
     * into a Row and extracts from that object which is a relatively
     * expensive operation, including full Avro deserialization.
     */
    public byte[] extractIndexKey(byte[] key,
                                  byte[] data,
                                  boolean keyOnly) {
        RowImpl row = table.createRowFromBytes(key, data, keyOnly);
        if (row != null) {
            return serializeIndexKey(row, false, 0);
        }
        return null;
    }

    /**
     * Extracts multiple index keys from a single record.  This is used if
     * one of the indexed fields is an array.  Only one array is allowed
     * in an index.
     *
     * @param key the key bytes
     *
     * @param data the row's data bytes
     *
     * @param keyOnly true if the index only uses key fields.  This
     * optimizes deserialization.
     *
     * @return a List of byte[] serializations of index keys or null if there
     * is no entry associated with the row, or the row does not match a
     * table record.
     *
     * While not likely it is possible that the record is not actually  a
     * table record and the key pattern happens to match.  Such records
     * will fail to be deserialized and throw an exception.  Rather than
     * treating this as an error, silently ignore it.
     *
     * TODO: can this be done without reserializing to Row?  It'd be
     * faster but more complex.
     *
     * 1.  Deserialize to RowImpl
     * 2.  Find the array value and get its size
     * 3.  for each array entry (size), serialize a key using that entry
     */
    public List<byte[]> extractIndexKeys(byte[] key,
                                         byte[] data,
                                         boolean keyOnly) {

        RowImpl row = table.createRowFromBytes(key, data, keyOnly);
        if (row != null) {
            String arrayField = findArray();
            assert arrayField != null;

            ArrayValueImpl fv =
                (ArrayValueImpl) row.getComplex(new IndexField(arrayField));
            if (fv == null || fv.isNull()) {
                return null;
            }

            int arraySize = fv.size();
            ArrayList<byte[]> returnList = new ArrayList<byte[]>(arraySize);
            for (int i = 0; i < arraySize; i++) {
                byte[] serKey = serializeIndexKey(row, false, i);

                /*
                 * It should not be possible for this to be null because
                 * it is not possible to add null values to arrays, but
                 * a bit of paranoia cannot hurt.
                 */
                if (serKey != null) {
                    returnList.add(serKey);
                }
            }
            return returnList;
        }
        return null;
    }

    public void toJsonNode(ObjectNode node) {
        node.put("name", name);
        node.put("description", description);
        ArrayNode fieldArray = node.putArray("fields");
        for (String s : fields) {
            fieldArray.add(s);
        }
    }

    /**
     * Validate that the name, fields, and types of the index match
     * the table.  This also initializes the (transient) list of index fields in
     * indexFields, so that member must not be used in validate() itself.
     *
     * This method must only be called from the constructor.  It is not
     * synchronized and changes internal state.
     */
    private void validate() {
        TableImpl.validateComponent(name, false);
        boolean hasArray = false;
        if (fields.isEmpty()) {
            throw new IllegalCommandException
                ("Index requires at least one field");
        }

        assert indexFields == null;

        indexFields = new ArrayList<IndexField>(fields.size());

        for (String field : fields) {
            if (field == null || field.length() == 0) {
                throw new IllegalCommandException
                    ("Invalid (null or empty) index field name");
            }
            IndexField ifield = new IndexField(field);

            /*
             * This call handles indexes into complex, nested types.
             */
            FieldDefImpl def = findIndexField(ifield);
            if (def == null) {
                throw new IllegalCommandException
                    ("Index field not found in table: " + field);
            }
            if (!def.isValidIndexField()) {
                throw new IllegalCommandException
                    ("Field type is not valid in an index: " +
                     def.getType() + ", field name: " + field);
            }

            /*
             * The check for arrays needs to consider all fields as well as
             * fields that reference into complex types.  An array may occur
             * at any point in the navigation path (first, interior, leaf).
             *
             * The call to containsArray() will set the array state in the
             * IndexField.
             */
            boolean fieldHasArray = containsArray(ifield);
            if (fieldHasArray) {
                if (hasArray) {
                    throw new IllegalCommandException
                        ("Indexes may contain only one array field");
                }
                hasArray = true;
            }
            if (indexFields.contains(ifield)) {
                throw new IllegalCommandException
                    ("Index already contains the field: " + field);
            }
            indexFields.add(ifield);
        }
        assert fields.size() == indexFields.size();
        table.checkForDuplicateIndex(this);
    }

    @Override
    public String toString() {
        return "Index[" + name + ", " + table.getId() + ", " + status + "]";
    }

    /**
     * Serialize the index fields from the RecordValueImpl argument.
     * Fields are extracted in index order.  It is assumed that the caller has
     * validated the record and that if it is an IndexKey that user-provided
     * fields are correct and in order.
     *
     * @param record the record to extract.  This may be an IndexKeyImpl or
     * RowImpl.  In both cases the caller can vouch for the validity of the
     * object.
     *
     * @param allowPartial if true then partial keys can be serialized.  This is
     * the case for client-based keys.  If false, partial keys result in
     * returning null.  This is the server side key extraction path.
     *
     * @param arrayIndex will be 0 if not doing an array lookup, or if the
     * desired array index is actually 0.  For known array lookups it may be
     * >0.
     *
     * @return the serialized index key or null if the record cannot
     * be serialized.
     *
     * These are conditions that will cause serialization to fail:
     * 1.  The record has a null values in one of the index keys
     * 2.  An index key field contains a map and the record does not
     * have a value for the indexed map key value
     */
    byte[] serializeIndexKey(RecordValueImpl record, boolean allowPartial,
                             int arrayIndex) {
        TupleOutput out = null;
        try {
            out = new TupleOutput();
            for (IndexField field : getIndexFields()) {
                FieldValue val = record.findField(field.iterator(), arrayIndex);
                FieldDefImpl def = findIndexField(field);
                assert def != null; /* can't happen */

                /*
                 * If the target field is an array use its type, which must be
                 * simple, and indexable.
                 */
                if (val != null && def.getType() == FieldDef.Type.ARRAY) {
                    def = (FieldDefImpl) ((ArrayDefImpl)def).getElement();
                    val = ((ArrayValueImpl)val).get(arrayIndex);
                }

                if (val == null) {

                    /* If the key must be fully present, fail */
                    if (!allowPartial) {
                        return null;
                    }

                    /* A partial key, done with fields */
                    break;
                }

                /*
                 * If any values are null it is not possible to serialize the
                 * index key so there is no entry for this index.
                 */
                if (val.isNull()) {
                    return null;
                }

                switch (def.getType()) {
                case INTEGER:
                    out.writeSortedPackedInt(val.asInteger().get());
                    break;
                case STRING:
                    out.writeString(val.asString().get());
                    break;
                case LONG:
                    out.writeSortedPackedLong(val.asLong().get());
                    break;
                case DOUBLE:
                    out.writeSortedDouble(val.asDouble().get());
                    break;
                case FLOAT:
                    out.writeSortedFloat(val.asFloat().get());
                    break;
                case ENUM:
                    /* enumerations are sorted by declaration order */
                    out.writeSortedPackedInt(val.asEnum().getIndex());
                    break;
                case ARRAY:
                case BINARY:
                case BOOLEAN:
                case FIXED_BINARY:
                case MAP:
                case RECORD:
                    throw new IllegalStateException
                        ("Type not supported in indexes: " +
                         def.getType());
                }
            }
            return (out.size() != 0 ? out.toByteArray() : null);
        } finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException ioe) {
            }
        }
    }

    /**
     * This is the version used by most client-based callers.  In this case
     * the key may be partially specified.
     *
     * @return the serialized index key or null if the record cannot
     * be serialized (e.g. it has null values).
     */
    public byte[] serializeIndexKey(IndexKeyImpl record) {
        return serializeIndexKey(record, true, 0);
    }

    /**
     * Deserialize an index key into IndexKey.  The caller will also have
     * access to the primary key bytes which can be turned into a PrimaryKey
     * and combined with the IndexKey for the returned KeyPair.
     *
     * Arrays -- if there is an array index the index key returned will
     * be the serialized value of a single array entry and not the array
     * itself. This value needs to be deserialized back into a single-value
     * array.
     *
     * @param data the bytes
     * @param partialOK true if not all fields must be in the data stream.
     */
    public IndexKeyImpl rowFromIndexKey(byte[] data, boolean partialOK) {
        IndexKeyImpl ikey = createIndexKey();
        TupleInput input = null;

        try {
            input = new TupleInput(data);
            for (IndexField field : getIndexFields()) {
                if (input.available() <= 0) {
                    break;
                }
                FieldDef def = findIndexField(field);
                assert def != null;
                switch (def.getType()) {
                case INTEGER:
                    ikey.putComplex(field, FieldDef.Type.INTEGER,
                                    input.readSortedPackedInt());
                    break;
                case STRING:
                    ikey.putComplex(field, FieldDef.Type.STRING, input.readString());
                    break;
                case LONG:
                    ikey.putComplex(field, FieldDef.Type.LONG,
                                    input.readSortedPackedLong());
                    break;
                case DOUBLE:
                    ikey.putComplex(field, FieldDef.Type.DOUBLE,
                                    input.readSortedDouble());
                    break;
                case FLOAT:
                    ikey.putComplex(field, FieldDef.Type.FLOAT,
                                    input.readSortedFloat());
                    break;
                case ENUM:
                    ikey.putComplex(field, FieldDef.Type.ENUM,
                                    input.readSortedPackedInt());
                    break;
                case ARRAY:

                    /* the data is not used by the array constructor */
                    ikey.putComplex(field, FieldDef.Type.ARRAY, null);
                    ArrayValueImpl array =
                        (ArrayValueImpl) ikey.getComplex(field);
                    readArrayElement(array, input);
                    break;
                case BINARY:
                case BOOLEAN:
                case FIXED_BINARY:
                case MAP:
                case RECORD:
                    throw new IllegalStateException
                        ("Type not supported in indexes: " +
                         def.getType());
                }
            }
            if (!partialOK && (ikey.numValues() != fields.size())) {
                throw new IllegalStateException
                    ("Missing fields from index data for index " +
                     getName() + ", expected " +
                     fields.size() + ", received " + ikey.numValues());
            }
            return ikey;
        } finally {
            try {
                if (input != null) {
                    input.close();
                }
            } catch (IOException ioe) {
            }
        }
    }

    private void readArrayElement(ArrayValueImpl array,
                                  TupleInput input) {
        switch (array.getDefinition().getElement().getType()) {
        case INTEGER:
            array.add(input.readSortedPackedInt());
            break;
        case STRING:
            array.add(input.readString());
            break;
        case LONG:
            array.add(input.readSortedPackedLong());
            break;
        case DOUBLE:
            array.add(input.readSortedDouble());
            break;
        case FLOAT:
            array.add(input.readSortedFloat());
            break;
        case ENUM:
            array.addEnum(input.readSortedPackedInt());
            break;
        default:
            throw new IllegalStateException("Type not supported in indexes: ");
        }
    }

    /**
     * Does a direct comparison of the IndexField to the existing fields to
     * look for duplicates.
     */
    boolean containsField(IndexField indexField) {
        for (IndexField iField : getIndexFields()) {
            if (iField.equals(indexField)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks to see if the index contains the *single* named field.
     * For simple types this is a simple contains operation.
     *
     * For complex types this needs to validate for a put of a complex
     * type that *may* contain an indexed field.
     * Validation of such fields must be done later.
     *
     * In the case of a nested field name with dot-separated names,
     * this code simply checks that fieldName is one of the components of
     * the complex field (using String.contains()).
     */
    boolean containsField(String fieldName) {
        String fname = fieldName.toLowerCase();

        for (IndexField indexField : getIndexFields()) {
            if (indexField.isComplex()) {
                if (indexField.getFieldName().contains(fname)) {
                    return true;
                }

            } else {
                if (indexField.getFieldName().equals(fname)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Locates the named field within the table's hierarchy.  The field
     * may be a simple, top-level field, or it may be in dot notation,
     * specifying a field in a nested type (record, map, array of (map|array)).
     * The ultimate field must be an indexable type.  That is checked in the
     * caller.
     *
     * When called internally using an already-validated IndexImpl and a field
     * that is known to exist, this method cannot fail to return an object.
     * When called during validation or with a field name passed from a user
     * (e.g. createFieldRange()) it can return null.
     */
    FieldDefImpl findIndexField(IndexField field) {

        Iterator<String> iter = field.iterator();
        FieldDefImpl def = (FieldDefImpl) table.getField(iter.next());

        if (def == null || !field.isComplex()) {
            return def;
        }

        /*
         * Call the FieldDef itself to navigate the names.
         */
        assert iter.hasNext();
        return def.findField(iter);
    }

    IndexField createIndexField(String fieldName) {
        return new IndexField(fieldName);
    }

    /**
     * Encapsulates a single field in an index, which may be simple or
     * complex.  Simple fields have a single component, fields that navigate
     * into nested fields have multiple components.
     *
     * Field names are case-insensitive, so strings are stored lower-case to
     * simplify case-insensitive comparisons.
     */
    class IndexField {
        final private String fieldName;
        final private List<String> fieldComponents;
        final private boolean isComplex;
        private String arrayPath;

        private IndexField(String field) {
            fieldName = field.toLowerCase();
            fieldComponents = parseComplexFieldName(fieldName);
            isComplex = (fieldComponents.size() > 1);
        }

        final boolean isComplex() {
            return isComplex;
        }

        final String getFieldName() {
            return fieldName;
        }

        final List<String> getComponents() {
            return fieldComponents;
        }

        Iterator<String> iterator() {
            return fieldComponents.iterator();
        }

        private boolean hasArray() {
            return arrayPath != null;
        }

        private String getArrayPath() {
            return arrayPath;
        }

        private void setArrayPath(String path) {
            arrayPath = path;
        }

        /**
         * Returns the FieldDef associated with the first (and maybe only)
         * component of the index field.
         */
        private FieldDefImpl getFirstDef() {
            return (FieldDefImpl) table.getField(fieldComponents.get(0));
        }

        @Override
        public String toString() {
            return fieldName;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof IndexField) {
                IndexField other = (IndexField) obj;
                if (getComponents().size() == other.getComponents().size()) {
                    Iterator<String> it = iterator();
                    Iterator<String> otherIt = other.iterator();
                    while (it.hasNext()) {
                        if (!it.next().equals(otherIt.next())) {
                            return false;
                        }
                    }
                return true;
                }
            }
            return false;
        }

        @Override
        public int hashCode() {
            int hash = 0;
            for (String s : fieldComponents) {
                hash += s.hashCode();
            }
            return hash;
        }

        /**
         * Returns a list of field names components in a complex field name,
         * or a single name if the field name is not complex (this is for
         * simplicity in use).
         *
         * FUTURE: maybe handle quoting of map field names.
         */
        private List<String> parseComplexFieldName(String fname) {
            List<String> list = new ArrayList<String>();
            StringBuilder sb = new StringBuilder();

            for (char ch : fname.toCharArray()) {
                if (ch == '.') {
                    if (sb.length() == 0) {
                        throw new IllegalArgumentException
                            ("Malformed field name: " + fname);
                    }
                    list.add(sb.toString());
                    sb.delete(0, sb.length());
                } else {
                    sb.append(ch);
                }
            }

            if (sb.length() > 0) {
                list.add(sb.toString());
            }
            return list;
        }
    }
}
