/*-
 *
 *  This file is part of Oracle NoSQL Database
 *  Copyright (C) 2011, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 * If you have received this file as part of Oracle NoSQL Database the
 * following applies to the work as a whole:
 *
 *   Oracle NoSQL Database server software is free software: you can
 *   redistribute it and/or modify it under the terms of the GNU Affero
 *   General Public License as published by the Free Software Foundation,
 *   version 3.
 *
 *   Oracle NoSQL Database is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Affero General Public License for more details.
 *
 * If you have received this file as part of Oracle NoSQL Database Client or
 * distributed separately the following applies:
 *
 *   Oracle NoSQL Database client software is free software: you can
 *   redistribute it and/or modify it under the terms of the Apache License
 *   as published by the Apache Software Foundation, version 2.0.
 *
 * You should have received a copy of the GNU Affero General Public License
 * and/or the Apache License in the LICENSE file along with Oracle NoSQL
 * Database client or server distribution.  If not, see
 * <http://www.gnu.org/licenses/>
 * or
 * <http://www.apache.org/licenses/LICENSE-2.0>.
 *
 * An active Oracle commercial licensing agreement for this product supersedes
 * these licenses and in such case the license notices, but not the copyright
 * notice, may be removed by you in connection with your distribution that is
 * in accordance with the commercial licensing terms.
 *
 * For more information please contact:
 *
 * berkeleydb-info_us@oracle.com
 *
 */

package oracle.kv.impl.api.table;

import java.util.ArrayList;
import java.util.List;

import oracle.kv.impl.api.table.IndexImpl.IndexField;
import oracle.kv.table.FieldDef;
import oracle.kv.table.FieldValue;
import oracle.kv.table.Index;
import oracle.kv.table.IndexKey;

public class IndexKeyImpl extends RecordValueImpl implements IndexKey {
    private static final long serialVersionUID = 1L;
    final IndexImpl index;

    /**
     * The RecordDef associated with an IndexKeyImpl is that of its table.
     */
    IndexKeyImpl(IndexImpl index) {
        super(new RecordDefImpl(index.getTableImpl().getName(),
                                index.getTableImpl().getFieldMap()));
        this.index = index;
    }

    private IndexKeyImpl(IndexKeyImpl other) {
        super(other);
        this.index = other.index;
    }

    /**
     * Return the Index associated with this key
     */
    @Override
    public Index getIndex() {
        return index;
    }

    @Override
    public IndexKeyImpl clone() {
        return new IndexKeyImpl(this);
    }

    @Override
    FieldDefImpl validateNameAndType(String name,
                                     FieldDef.Type type) {
        FieldDefImpl def = super.validateNameAndType(name, type);
        if (!index.containsField(name)) {
            throw new IllegalArgumentException
                ("Field is not part of Index: " + name);
        }
        return def;
    }

    @Override
    public IndexKey asIndexKey() {
        return this;
    }

    @Override
    public boolean isIndexKey() {
        return true;
    }

    @Override
    public boolean equals(Object other) {
        if (super.equals(other)) {
            return other instanceof IndexKeyImpl;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    int getNumFields() {
        return index.getFieldsInternal().size();
    }

    @Override
    FieldDef getField(String fieldName) {
        FieldDef def = getDefinition().getField(fieldName);
        if (def != null && index.containsField(fieldName)) {
            return def;
        }
        return null;
    }

    @Override
    FieldMapEntry getFieldMapEntry(String fieldName) {
        FieldMapEntry fme = getDefinition().getFieldMapEntry(fieldName, false);
        if (fme != null && index.containsField(fieldName)) {
            return fme;
        }
        return null;
    }

    @Override
    public List<String> getFields() {
        return index.getFields();
    }

    /**
     * Gets the fields that are associated with the index.  This method is
     * called by copy methods and other functions that only expect the top-level
     * fields and not any nested fields, so in the case of a complex field,
     * only return the first component.
     */
    @Override
    protected List<String> getFieldsInternal() {
        List<IndexField> indexFields = index.getIndexFields();
        List<String> list = new ArrayList<String>(indexFields.size());
        for (IndexField indexField : indexFields)  {
            list.add(indexField.getComponents().get(0));
        }
        return list;
    }

    /**
     * IndexKey cannot contain null values
     */
    @Override
    public IndexKey putNull(String name) {
        throw new IllegalArgumentException
            ("IndexKey may not contain null values");
    }

    /**
     * Creates a record with its index field set for validation.
     */
    @Override
    public RecordValueImpl putRecord(String fieldName) {
        RecordValueImpl record = super.putRecord(fieldName);
        record.setIndex(index);
        return record;
    }

    /**
     * Creates a map with its index field set for validation.
     */
    @Override
    public MapValueImpl putMap(String fieldName) {
        MapValueImpl map = super.putMap(fieldName);
        map.setIndex(index);
        return map;
    }

    /**
     * Creates an array with its index field set for validation.
     */
    @Override
    public ArrayValueImpl putArray(String fieldName) {
        ArrayValueImpl array = super.putArray(fieldName);
        array.setIndex(index);
        return array;
    }

    public int getKeySize() {
        return index.serializeIndexKey(this).length;
    }

    TableImpl getTable() {
        return index.getTableImpl();
    }

    void putComplex(String fieldName, FieldDef.Type type, FieldValue value) {
        putComplex(index.createIndexField(fieldName), type, value);
    }

    /**
     * Puts a potentially nested value into an index key.  This exists for the
     * CLI to allow it to put nested values by name vs via JSON.
     */
    public void putComplexField(String fieldName, FieldValue value) {
        putComplex(fieldName, value.getType(), value);
    }

    /**
     * Finds a potentially nested FieldDef by name.  This exists for the CLI to
     * allow it to put nested values by name vs via JSON.
     */
    public FieldDefImpl findFieldDefinition(String fieldName) {
        FieldDefImpl def =
            index.findIndexField(index.createIndexField(fieldName));
        if (def == null) {
            throw new IllegalArgumentException
                ("Field does not exist in the index: " + fieldName);
        }
        return def;
    }

    /**
     * Validate the index key.  Rules:
     * 1. Fields must be in the index
     * 2. Fields must be specified in order.  If a field "to the right"
     * in the index definition is set, all fields to its "left" must also
     * be present.
     *
     * If the field is in a nested type this code will fetch the actual "leaf"
     * FieldValue and validate.
     */
    @Override
    void validate() {
        int numFound = 0;
        int i = 0;
        for (IndexField field : index.getIndexFields()) {
            FieldValueImpl val = getComplex(field);
            if (val != null) {
                if (i != numFound) {
                    throw new IllegalArgumentException
                        ("IndexKey is missing fields more significant than" +
                         " field: " + field);
                }
                if (!index.containsField(field)) {
                    throw new IllegalArgumentException
                        ("Field is not part of Index: " + field);
                }
                if (val.isNull()) {
                    throw new IllegalArgumentException
                        ("Field value is null, which is invalid for " +
                         "IndexKey: " + field);
                }

                /*
                 * If the field is an array it must have a single entry.
                 */
                if (val.isArray() && val.asArray().size() != 1) {
                    throw new IllegalArgumentException
                        ("Arrays used in index keys must contain a single " +
                         "element");
                }
                ++numFound;
            }
            ++i;
        }

        /*
         * numValues() counts all values in the record that are set, descending
         * into complex types as well.  This ensures that there are no extra
         * fields in an IndexKey that do not belong.  Unfortunately this check
         * won't occur until after the IndexKey has been created.  Bad top-level
         * fields are validated when they are set, but bad nested fields aren't
         * found until use of the IndexKey.
         */
        if (numFound != numValues()) {
            throw new IllegalArgumentException
                ("IndexKey contains a field that is not part of the Index");
        }
    }

    public IndexImpl getIndexImpl() {
        return index;
    }

    /**
     * Return true if all fields in the index are specified.
     */
    boolean isComplete() {
        return (size() == getNumFields());
    }

    /**
     * This function behaves like adding "one" to the entire index key.  That
     * is, it increments the least significant field but if that field "wraps"
     * in that it's already at its maximum in terms of data type, such as
     * Integer.MAX_VALUE then increment the next more significant field and
     * set that field to its minimum value.
     *
     * If the value would wrap and there are no more significant fields then
     * return false, indicating that the "next" value is actually the end
     * of the index, period.
     *
     * This code is used to implement inclusive/exclusive semantics.
     */
    public boolean incrementIndexKey() {
        FieldValue[] values = new FieldValue[index.numFields()];
        int fieldIndex = 0;
        List<IndexField> indexFields = index.getIndexFields();
        for (IndexField field : indexFields) {
            values[fieldIndex] = getComplex(field);
            if (values[fieldIndex] == null) {
                break;
            }
            fieldIndex++;
        }

        /*
         * At least one field must exist.  Assert that and move back to the
         * target field.
         */
        assert fieldIndex > 0;
        --fieldIndex;

        /*
         * Increment and reset.  If the current field returns null, indicating
         * that it will wrap its value, set it to its minimum value and move to
         * the next more significant field.  If there are none, return false
         * indicating that there are no larger keys in the index that match the
         * key.
         */
        FieldValueImpl fvi = ((FieldValueImpl)values[fieldIndex]).getNextValue();
        while (fvi == null) {
            fvi = ((FieldValueImpl)values[fieldIndex]).getMinimumValue();
            putComplex(indexFields.get(fieldIndex), fvi.getType(), fvi);

            /*
             * Move to next more significant field if it exists
             */
            --fieldIndex;
            if (fieldIndex >= 0) {
                fvi = ((FieldValueImpl)values[fieldIndex]).getNextValue();
            } else {
                /*
                 * Failed to increment
                 */
                return false;
            }
        }
        assert fvi != null && fieldIndex >= 0;
        putComplex(indexFields.get(fieldIndex), fvi.getType(), fvi);
        return true;
    }
}
