/*-
 *
 *  This file is part of Oracle NoSQL Database
 *  Copyright (C) 2011, 2014 Oracle and/or its affiliates.  All rights reserved.
 *
 * If you have received this file as part of Oracle NoSQL Database the
 * following applies to the work as a whole:
 *
 *   Oracle NoSQL Database server software is free software: you can
 *   redistribute it and/or modify it under the terms of the GNU Affero
 *   General Public License as published by the Free Software Foundation,
 *   version 3.
 *
 *   Oracle NoSQL Database is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Affero General Public License for more details.
 *
 * If you have received this file as part of Oracle NoSQL Database Client or
 * distributed separately the following applies:
 *
 *   Oracle NoSQL Database client software is free software: you can
 *   redistribute it and/or modify it under the terms of the Apache License
 *   as published by the Apache Software Foundation, version 2.0.
 *
 * You should have received a copy of the GNU Affero General Public License
 * and/or the Apache License in the LICENSE file along with Oracle NoSQL
 * Database client or server distribution.  If not, see
 * <http://www.gnu.org/licenses/>
 * or
 * <http://www.apache.org/licenses/LICENSE-2.0>.
 *
 * An active Oracle commercial licensing agreement for this product supersedes
 * these licenses and in such case the license notices, but not the copyright
 * notice, may be removed by you in connection with your distribution that is
 * in accordance with the commercial licensing terms.
 *
 * For more information please contact:
 *
 * berkeleydb-info_us@oracle.com
 *
 */

package oracle.kv.impl.map;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import oracle.kv.Key;
import oracle.kv.impl.topo.PartitionId;

/**
 * A hash based implementation used to distribute keys across partitions.
 */
public class HashKeyToPartitionMap implements KeyToPartitionMap {

    private static final long serialVersionUID = 1L;

    final BigInteger nPartitions;

    transient DigestCache digestCache = new DigestCache();

    public HashKeyToPartitionMap(int nPartitions) {
        super();
        this.nPartitions = new BigInteger(Integer.toString(nPartitions));
    }

    @Override
    public int getNPartitions() {
        return nPartitions.intValue();
    }

    @Override
    public PartitionId getPartitionId(byte[] keyBytes) {
        if (digestCache == null) {
            digestCache = new DigestCache();
        }
        /* Clone one for use by this thread. */
        final MessageDigest md = digestCache.get();

        /* Digest Key major path. */
        md.update(keyBytes, 0, Key.getMajorPathLength(keyBytes));

        final BigInteger index = new BigInteger(md.digest()).mod(nPartitions);
        return new PartitionId(index.intValue() + 1);
    }

    /**
     * Implements a per-thread cache using a thread local, to mitigate the cost
     * of calling MessageDigest.getInstance("MD5").
     */
    static class DigestCache extends ThreadLocal<MessageDigest> {

        /** Create the message digest. */
        @Override
        protected MessageDigest initialValue() {
            try {
                return MessageDigest.getInstance("MD5");
            } catch (NoSuchAlgorithmException e) {
                throw new IllegalStateException("MD5 algorithm unavailable");
            }
        }

        /** Reset the message digest before returning. */
        @Override
        public MessageDigest get() {
            final MessageDigest md = super.get();
            md.reset();
            return md;
        }
    }
}
