/*
 * Decompiled with CFR 0.152.
 */
package oracle.nosql.driver.iam;

import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import java.net.URI;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import javax.net.ssl.SSLException;
import javax.security.auth.RefreshFailedException;
import javax.security.auth.Refreshable;
import oracle.nosql.driver.NoSQLHandleConfig;
import oracle.nosql.driver.SecurityInfoNotReadyException;
import oracle.nosql.driver.httpclient.HttpClient;
import oracle.nosql.driver.iam.CertificateSupplier;
import oracle.nosql.driver.iam.FederationRequestHelper;
import oracle.nosql.driver.iam.SessionKeyPairSupplier;
import oracle.nosql.driver.iam.Utils;
import oracle.nosql.driver.util.CheckNull;

class SecurityTokenSupplier {
    private String tenantId;
    private int timeoutMS;
    private URI federationURL;
    private HttpClient federationClient;
    private final CertificateSupplier leafCertSupplier;
    private final Set<CertificateSupplier> intermediateCertificateSuppliers;
    private final SessionKeyPairSupplier keyPairSupplier;
    private final String purpose;
    private long minTokenLifetime;
    private final Logger logger;

    SecurityTokenSupplier(String federationEndpoint, String tenantId, CertificateSupplier leafCertSupplier, Set<CertificateSupplier> interCertificateSuppliers, SessionKeyPairSupplier keyPairSupplier, String purpose, int timeoutMS, Logger logger) {
        this.federationURL = URI.create(federationEndpoint + "/v1/x509");
        CheckNull.requireNonNullIAE(leafCertSupplier, "Certificate supplier cannot be null");
        this.leafCertSupplier = leafCertSupplier;
        CheckNull.requireNonNullIAE(keyPairSupplier, "Keypair supplier cannot be null");
        this.keyPairSupplier = keyPairSupplier;
        this.intermediateCertificateSuppliers = interCertificateSuppliers;
        CheckNull.requireNonNullIAE(tenantId, "Tenant id cannot be null");
        this.tenantId = tenantId;
        CheckNull.requireNonNullIAE(purpose, "Purpose cannot be null");
        this.purpose = purpose;
        this.timeoutMS = timeoutMS;
        this.logger = logger;
    }

    String getSecurityToken() {
        return this.refreshAndGetTokenInternal();
    }

    void setMinTokenLifetime(long lifetimeMS) {
        this.minTokenLifetime = lifetimeMS;
    }

    void close() {
        if (this.federationClient != null) {
            this.federationClient.shutdown();
        }
    }

    synchronized void prepare(NoSQLHandleConfig config) {
        if (this.federationClient == null) {
            this.federationClient = SecurityTokenSupplier.buildHttpClient(this.federationURL, config.getSslContext(), config.getSSLHandshakeTimeout(), this.logger);
        }
    }

    private static HttpClient buildHttpClient(URI endpoint, SslContext sslCtx, int sslHandshakeTimeout, Logger logger) {
        String scheme = endpoint.getScheme();
        if (scheme == null) {
            throw new IllegalArgumentException("Unable to find URL scheme, invalid URL " + endpoint.toString());
        }
        if (scheme.equalsIgnoreCase("http")) {
            return HttpClient.createMinimalClient(endpoint.getHost(), endpoint.getPort(), null, 0, "FederationClient", logger);
        }
        if (sslCtx == null) {
            try {
                sslCtx = SslContextBuilder.forClient().build();
            }
            catch (SSLException se) {
                throw new IllegalStateException("Unable to build SSL context for http client", se);
            }
        }
        return HttpClient.createMinimalClient(endpoint.getHost(), 443, sslCtx, sslHandshakeTimeout, "FederationClient", logger);
    }

    private synchronized String refreshAndGetTokenInternal() {
        this.keyPairSupplier.refreshKeys();
        if (this.leafCertSupplier instanceof Refreshable) {
            try {
                ((Refreshable)((Object)this.leafCertSupplier)).refresh();
            }
            catch (RefreshFailedException rfe) {
                throw new IllegalStateException("Can't refresh the leaf certification!", rfe);
            }
            String newTenantId = Utils.getTenantId(this.leafCertSupplier.getCertificateAndKeyPair().getCertificate());
            Utils.logTrace(this.logger, "Tenant id in refreshed certificate " + newTenantId);
            if (!this.tenantId.equals(newTenantId)) {
                throw new IllegalArgumentException("The tenant id should never be changed in certificate");
            }
        }
        for (CertificateSupplier supplier : this.intermediateCertificateSuppliers) {
            if (!(supplier instanceof Refreshable)) continue;
            try {
                ((Refreshable)((Object)supplier)).refresh();
            }
            catch (RefreshFailedException e) {
                throw new IllegalStateException("Can't refresh the intermediate certification!", e);
            }
        }
        return this.getSecurityTokenFromIAM();
    }

    private String getSecurityTokenFromIAM() {
        KeyPair keyPair = this.keyPairSupplier.getKeyPair();
        CheckNull.requireNonNullIAE(keyPair, "Keypair for session not provided");
        RSAPublicKey publicKey = (RSAPublicKey)keyPair.getPublic();
        CheckNull.requireNonNullIAE(publicKey, "Public key not present");
        CertificateSupplier.X509CertificateKeyPair certificateAndKeyPair = this.leafCertSupplier.getCertificateAndKeyPair();
        CheckNull.requireNonNullIAE(certificateAndKeyPair, "Certificate and key pair not present");
        X509Certificate leafCertificate = certificateAndKeyPair.getCertificate();
        CheckNull.requireNonNullIAE(leafCertificate, "Leaf certificate not present");
        CheckNull.requireNonNullIAE(certificateAndKeyPair.getKey(), "Leaf certificate's private key not present");
        try {
            HashSet<String> intermediateStrings = null;
            if (this.intermediateCertificateSuppliers != null && this.intermediateCertificateSuppliers.size() > 0) {
                intermediateStrings = new HashSet<String>();
                for (CertificateSupplier supplier : this.intermediateCertificateSuppliers) {
                    CertificateSupplier.X509CertificateKeyPair pair = supplier.getCertificateAndKeyPair();
                    if (pair == null || pair.getCertificate() == null) continue;
                    intermediateStrings.add(Utils.base64EncodeNoChunking(pair));
                }
            }
            String securityToken = this.getSecurityTokenInternal(Utils.base64EncodeNoChunking(publicKey), Utils.base64EncodeNoChunking(certificateAndKeyPair), intermediateStrings);
            SecurityToken st = new SecurityToken(securityToken, this.keyPairSupplier);
            st.validate(this.minTokenLifetime, this.logger);
            return st.getSecurityToken();
        }
        catch (Exception e) {
            throw new SecurityInfoNotReadyException(e.getMessage(), e);
        }
    }

    private String getSecurityTokenInternal(String publicKey, String leafCertificate, Set<String> interCerts) {
        String body = FederationRequestHelper.getFederationRequestBody(publicKey, leafCertificate, interCerts, this.purpose);
        Utils.logTrace(this.logger, "Federation request body " + body);
        return FederationRequestHelper.getSecurityToken(this.federationClient, this.federationURL, this.timeoutMS, this.tenantId, this.leafCertSupplier.getCertificateAndKeyPair(), body, this.logger);
    }

    static class SecurityToken {
        private final SessionKeyPairSupplier keyPairSupplier;
        private final String securityToken;
        private final Map<String, String> tokenClaims;
        private final long creationTime;
        private long expiryMS = -1L;

        SecurityToken(String token, SessionKeyPairSupplier supplier) {
            this.securityToken = token;
            this.keyPairSupplier = supplier;
            this.creationTime = System.currentTimeMillis();
            this.tokenClaims = Utils.parseToken(token);
        }

        String getSecurityToken() {
            return this.securityToken;
        }

        String getStringClaim(String key) {
            return this.tokenClaims.get(key);
        }

        long getCreationTime() {
            return this.creationTime;
        }

        long getExpiryMS() {
            return this.expiryMS;
        }

        void validate(long minTokenLifetime, Logger logger) {
            if (this.tokenClaims == null) {
                throw new IllegalArgumentException("No security token cached");
            }
            if (this.expiryMS == -1L) {
                String exp = this.tokenClaims.get("exp");
                if (exp != null) {
                    this.expiryMS = Long.parseLong(exp) * 1000L;
                }
                if (exp == null || this.expiryMS == -1L) {
                    throw new IllegalArgumentException("No expiration info in token:\n" + this.securityToken);
                }
            }
            long tokenLifetime = this.getExpiryMS() - this.getCreationTime();
            Utils.logTrace(logger, "New security token: lifetime=" + tokenLifetime + ", expiry=" + this.getExpiryMS() + ", creation=" + this.getCreationTime());
            if (tokenLifetime < minTokenLifetime) {
                Utils.logTrace(logger, "Security token has shorter lifetimethan expected minimal token lifetime, token:\n" + this.getSecurityToken());
            }
            String modulus = this.tokenClaims.get("n");
            String exponent = this.tokenClaims.get("e");
            if (modulus == null) {
                throw new IllegalArgumentException("Invalid JWK, no modulus in token:\n" + this.securityToken);
            }
            if (exponent == null) {
                throw new IllegalArgumentException("Invalid JWK, no exponent in token:\n" + this.securityToken);
            }
            RSAPublicKey jwkRsa = Utils.toPublicKey(modulus, exponent);
            RSAPublicKey expected = (RSAPublicKey)this.keyPairSupplier.getKeyPair().getPublic();
            if (jwkRsa == null) {
                throw new IllegalArgumentException("Invalid JWK, unable to find public key in token:\n" + this.securityToken);
            }
            if (!this.isEqualPublicKey(jwkRsa, expected)) {
                throw new IllegalArgumentException("Invalid JWK, public key not match in token:\n" + this.securityToken);
            }
        }

        private boolean isEqualPublicKey(RSAPublicKey actual, RSAPublicKey expect) {
            if (actual == null || expect == null) {
                throw new IllegalArgumentException("Public key cannot be null");
            }
            String actualKey = Utils.base64EncodeNoChunking(actual);
            String expectKey = Utils.base64EncodeNoChunking(expect);
            return actualKey.equals(expectKey);
        }
    }

    public static interface SecurityTokenBasedProvider {
        public void setMinTokenLifetime(long var1);

        default public void prepare(NoSQLHandleConfig config) {
        }

        default public void close() {
        }
    }
}

