/**
 * Copyright (c) 2016, 2022, Oracle and/or its affiliates.  All rights reserved.
 * This software is dual-licensed to you under the Universal Permissive License (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl or Apache License 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose either license.
 */
package com.oracle.bmc.model;

import com.oracle.bmc.ClientRuntime;
import com.oracle.bmc.ServiceDetails;
import com.oracle.bmc.http.internal.RFC3339DateFormat;
import com.oracle.bmc.util.internal.StringUtils;
import java.util.Date;
import java.util.Map;

public class BmcException extends RuntimeException {
    /**
     * Name of the header that contains the request id.
     */
    public static final String OPC_REQUEST_ID_HEADER = "opc-request-id";

    /**
     * The HTTP status code.
     */
    private final int statusCode;

    /**
     * Service specific code returned. Null if the client timed out or failed to get a response from the service.
     */
    private final String serviceCode;

    /**
     * Flag to indicate that the request timed out. Status code and service code should not be used if this is true.
     */
    private final boolean timeout;

    /**
     * The opc-request-id header returned by the service that should be provided for support requests.
     *
     * If the client timed out or failed to get a response from the service, then this is the outbound request id,
     * i.e. either the value for the {@link BmcException#OPC_REQUEST_ID_HEADER} header set by the caller in an
     * invocation callback, or otherwise a value automatically generated by the SDK.
     */
    private final String opcRequestId;

    /**
     * Flag to indicate that the exception originated from the client and not from the service.
     * Status code and service code should not be used if this is true.
     */
    private final boolean isClientSide;

    /**
     * Details about the service of the Exception
     */
    private ServiceDetails serviceDetails;

    /**
     * A human-readable error string in English.
     * Optional field in localized error responses.
     */
    private final String originalMessage;

    /**
     * Template in ICU MessageFormat for the human-readable error string in English, but without the values replaced.
     * Optional field in localized error responses.
     */
    private final String originalMessageTemplate;

    /**
     * The values to be substituted into the originalMessageTemplate, expressed as a string-to-string map.
     * Optional field in localized error responses.
     */
    private final Map<String, String> messageArguments;

    public BmcException(
            int statusCode,
            String serviceCode,
            String message,
            String opcRequestId,
            Throwable cause) {
        super(message, cause);
        this.statusCode = statusCode;
        this.serviceCode = serviceCode;
        this.opcRequestId = opcRequestId;
        this.timeout = false;
        this.isClientSide = false;
        this.originalMessage = null;
        this.originalMessageTemplate = null;
        this.messageArguments = null;
    }

    public BmcException(
            int statusCode,
            String serviceCode,
            String message,
            String opcRequestId,
            Throwable cause,
            ServiceDetails serviceDetails) {
        super(message, cause);
        this.statusCode = statusCode;
        this.serviceCode = serviceCode;
        this.opcRequestId = opcRequestId;
        this.timeout = false;
        this.isClientSide = false;
        this.serviceDetails = serviceDetails;
        this.originalMessage = null;
        this.originalMessageTemplate = null;
        this.messageArguments = null;
    }

    public BmcException(int statusCode, String serviceCode, String message, String opcRequestId) {
        super(message);
        this.statusCode = statusCode;
        this.serviceCode = serviceCode;
        this.opcRequestId = opcRequestId;
        this.timeout = false;
        this.isClientSide = false;
        this.originalMessage = null;
        this.originalMessageTemplate = null;
        this.messageArguments = null;
    }

    public BmcException(
            int statusCode,
            String serviceCode,
            String message,
            String opcRequestId,
            ServiceDetails serviceDetails) {
        super(message);
        this.statusCode = statusCode;
        this.serviceCode = serviceCode;
        this.opcRequestId = opcRequestId;
        this.timeout = false;
        this.isClientSide = false;
        this.serviceDetails = serviceDetails;
        this.originalMessage = null;
        this.originalMessageTemplate = null;
        this.messageArguments = null;
    }

    public BmcException(boolean timeout, String message, Throwable cause, String opcRequestId) {
        super(message, cause);
        this.statusCode = -1;
        this.serviceCode = null;
        this.opcRequestId = opcRequestId;
        this.timeout = timeout;
        this.isClientSide = true;
        this.originalMessage = null;
        this.originalMessageTemplate = null;
        this.messageArguments = null;
    }

    public BmcException(
            int statusCode,
            String serviceCode,
            String message,
            String opcRequestId,
            ServiceDetails serviceDetails,
            String originalMessage,
            String originalMessageTemplate,
            Map<String, String> templateArguments) {
        super(message);
        this.statusCode = statusCode;
        this.serviceCode = serviceCode;
        this.opcRequestId = opcRequestId;
        this.timeout = false;
        this.isClientSide = false;
        this.serviceDetails = serviceDetails;
        this.originalMessage = originalMessage;
        this.originalMessageTemplate = originalMessageTemplate;
        this.messageArguments = templateArguments;
    }

    @Override
    public String getMessage() {
        String requestId =
                this.opcRequestId != null
                        ? " ("
                                + (isClientSide ? "outbound " : "")
                                + "opc-request-id: "
                                + this.opcRequestId
                                + ")"
                        : "";
        if (null != serviceDetails) {
            String targetService = serviceDetails.getServiceName();
            String timestamp = RFC3339DateFormat.formatRfc3339(new Date(), true);
            String clientVersion = ClientRuntime.getRuntime().getClientInfo();
            String errorTroubleshootingLink =
                    String.format(
                            "https://docs.oracle.com/en-us/iaas/Content/API/References/apierrors.htm#apierrors_%s__%s_%s",
                            statusCode,
                            statusCode,
                            StringUtils.isNotBlank(serviceCode) ? serviceCode.toLowerCase() : "");

            return String.format(
                    "Error returned by %s operation in %s service."
                            + "(%s, %s, %s) %s%s"
                            + "\nTimestamp: %s"
                            + "\nClient version: %s"
                            + "\nRequest Endpoint: %s"
                            + "\nTroubleshooting Tips: See %s for more information about resolving this error"
                            + "\nAlso see %s for details on this operation's requirements."
                            + "\nTo get more info on the failing request, you can enable debug level logs as mentioned in `Using SLF4J for Logging section` in https://docs.oracle.com/en-us/iaas/Content/API/SDKDocs/javasdkconfig.htm."
                            + "\nIf you are unable to resolve this %s issue, please contact Oracle support and provide them this full error message.",
                    serviceDetails.getOperationName(),
                    targetService,
                    statusCode,
                    serviceCode,
                    timeout,
                    super.getMessage(),
                    requestId,
                    timestamp,
                    clientVersion,
                    serviceDetails.getRequestEndpoint(),
                    errorTroubleshootingLink,
                    serviceDetails.getApiReferenceLink(),
                    targetService);
        } else {
            return String.format(
                    "(%s, %s, %s) %s%s",
                    statusCode,
                    serviceCode,
                    timeout,
                    super.getMessage(),
                    requestId);
        }
    }

    /**
     * The HTTP status code.
     */
    public int getStatusCode() {
        return this.statusCode;
    }

    /**
     * Service specific code returned. Null if the client timed out or failed to get a response from the service.
     */
    public String getServiceCode() {
        return this.serviceCode;
    }

    /**
     * Flag to indicate that the request timed out. Status code and service code should not be used if this is true.
     */
    public boolean isTimeout() {
        return this.timeout;
    }

    /**
     * The opc-request-id header returned by the service that should be provided for support requests.
     *
     * If the client timed out or failed to get a response from the service, then this is the outbound request id,
     * i.e. either the value for the {@link BmcException#OPC_REQUEST_ID_HEADER} header set by the caller in an
     * invocation callback, or otherwise a value automatically generated by the SDK.
     */
    public String getOpcRequestId() {
        return this.opcRequestId;
    }

    /**
     * Flag to indicate that the exception originated from the client and not from the service.
     * Status code and service code should not be used if this is true.
     */
    public boolean isClientSide() {
        return this.isClientSide;
    }

    /**
     * Gets the service details of the failed Exception. null in the case of client-side exceptions
     */
    public ServiceDetails getServiceDetails() {
        return this.serviceDetails;
    }

    /**
     * Gets the original message of the failed Exception. null in the case of non-localized responses.
     */
    public String getOriginalMessage() {
        return this.originalMessage;
    }

    /**
     * Gets the original message template of the failed Exception. null in the case of non-localized responses.
     */
    public String getOriginalMessageTemplate() {
        return this.originalMessageTemplate;
    }

    /**
     * Gets the template arguments of the failed Exception. null in the case of non-localized responses.
     */
    public Map<String, String> getMessageArguments() {
        return this.messageArguments;
    }
}
