package com.outr.mailgun

/** @see Email */
object Email {
  /** @see Addr */
  object Addr {
    /** Alternate constructor */
    def apply(address: String, name: String): Addr = new Addr(address, Some(name))
  }

  /**
    * A simple regex to do a quick check on an email address. I'm not
    * interested in being thorough here, its just a sanity check. The Mailgun
    * server will do more extensive checks.
    */
  private lazy val emailRegex = List(
    """^[a-zA-Z0-9.!#$%&’*+/=?^_`{|}~-]+""",
    """@""",
    """[a-zA-Z0-9-]+(?:\.[a-zA-Z0-9-]+)*$"""
  ).mkString.r

  /** An email address */
  case class Addr(address: String, name: Option[String] = None) {
    if (emailRegex.findFirstIn(address).isEmpty) {
      throw new IllegalArgumentException("Invalid email address: %s".format(address))
    }

    /** Returns this email address encoded as a string */
    def encode: String = name match {
      case Some(innerName) => "%s <%s>".format(innerName, address)
      case None => address
    }
  }

  /** Email content */
  trait Body {
    /** Returns this body as a map */
    def toMap: Map[String, String]
  }

  /** Text content */
  trait TextBody extends Body

  /** HTML Content */
  trait HtmlBody extends Body


  /** Uses text for the body of an email */
  def text(content: String): TextBody = new TextBody {
    override def toMap = Map("text" -> content)

    override def toString = "Text(%s)".format(content)
  }

  /** Uses html for the body of an email */
  def html(content: String): HtmlBody = new HtmlBody {
    override def toMap = Map("html" -> content)

    override def toString = "Html(%s)".format(content)
  }

  /** Uses text and html for the body of an email */
  def textAndHtml(textContent: String,
                  htmlContent: String): TextBody with HtmlBody = new TextBody with HtmlBody {
    override def toMap = Map("text" -> textContent, "html" -> htmlContent)

    override def toString = "Html(%s) Text(%s)".format(textContent, htmlContent)
  }
}

/** A specific email to be sent */
case class Email(to: Email.Addr,
                 from: Email.Addr,
                 subject: String,
                 body: Email.Body,
                 tag: Option[String] = None) {
  /** Construct from strings */
  def this(to: String, from: String, subject: String, body: String) = this(
    Email.Addr(to, None), Email.Addr(from, None),
    subject, Email.html(body))

  /** Converts this instance to a Map */
  def toMap: Map[String, String] = {
    Map(
      "to" -> to.encode,
      "from" -> from.encode,
      "subject" -> subject
    ) ++ body.toMap ++ (if (tag.isDefined) Map("tag" -> tag.get) else Map())
  }
}

