package com.outr.mailgun

import java.util.concurrent.atomic.AtomicReference

import com.roundeights.scalon.{nException, nObject, nParser}
import org.asynchttpclient.AsyncHandler.State
import org.asynchttpclient.{AsyncHandler, AsyncHttpClient, DefaultAsyncHttpClient, DefaultAsyncHttpClientConfig, HttpResponseBodyPart, HttpResponseHeaders, HttpResponseStatus, Request, RequestBuilder}

import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.Try

/**
  * Makes REST requests.
  */
private class Requestor(private val client: AsyncHttpClient)(implicit context: ExecutionContext) {
  /** Alternate constructor that puts together an async client */
  def this(timeout: Int, maxConnections: Int)(implicit context: ExecutionContext) = this(
    new DefaultAsyncHttpClient(
      new DefaultAsyncHttpClientConfig.Builder()
        .setMaxRedirects(3)
        .setRequestTimeout(timeout)
        .setMaxConnectionsPerHost(maxConnections)
        .build()
    )
  )

  /** Shutsdown the underlying client */
  def close() = client.close()

  /** Executes a request */
  private def request(req: Request): Future[nObject] = {
    val async = new Asynchronizer
    client.executeRequest(req, async)
    async.future
  }

  /** Executes a request */
  def request(url: String,
              headers: Map[String, String],
              params: Map[String, String]): Future[nObject] = {
    val builder = new RequestBuilder("POST").setUrl(url)
    headers.foreach(pair => builder.addHeader(pair._1, pair._2))
    params.foreach(pair => builder.addQueryParam(pair._1, pair._2))
    request(builder.build)
  }
}

/**
  * The asynchronous request handler that accumulates a request as it is
  * receied.
  */
private class Asynchronizer extends AsyncHandler[Unit] {
  /** The promise that will contain the output of this request */
  private val result = Promise[nObject]()

  /** The error status, if one was encountered */
  private val status = new AtomicReference[Option[HttpResponseStatus]](None)

  /** Collects the body of the request as it is received */
  private val body = new StringBuffer

  /** Returns the future from this asynchronizer */
  def future = result.future

  override def onThrowable(t: Throwable): Unit = result.failure(t)

  override def onBodyPartReceived(
                                   bodyPart: HttpResponseBodyPart
                                 ): State = {
    body.append(new String(bodyPart.getBodyPartBytes, "UTF-8"))
    State.CONTINUE
  }

  override def onStatusReceived(responseStatus: HttpResponseStatus): State = {
    // Store non-success responses, as they indicate an error
    if (responseStatus.getStatusCode != 200) {
      status.set(Some(responseStatus))
    }
    State.CONTINUE
  }

  override def onHeadersReceived(headers: HttpResponseHeaders): State = State.CONTINUE

  override def onCompleted(): Unit = {
    status.get match {
      case None => result.complete(Try {
        nParser.jsonObj(body.toString)
      })
      case Some(s) => {
        try {
          val json = nParser.jsonObj(body.toString)
          result.failure(new MailSender.Error(json.str("message")))
        } catch {
          // If parsing the JSON fails, send a general request failure
          case _: nException => result.failure(new MailSender.Error("%d: %s".format(s.getStatusCode, s.getStatusText)))
          case err: Throwable => result.failure(err)
        }
      }
    }
  }
}