/**
 * Copyright 2012, OVH. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */
package com.ovh.ws.api;

public class OvhWsException extends Exception {

	private Type type;
	private Category category;

	public OvhWsException(Type type, String message) {
		this(type, message, null);
	}

	public OvhWsException(Type type, String message, Throwable cause) {
		super(message, cause);

		this.type = type;
		category = Category.fromInteger(type.getStatus());
	}

	public OvhWsException(int status, String message) {
		this(status, message, null);
	}

	public OvhWsException(int status, String message, Throwable cause) {
		super(message, cause);

		try {
			type = Type.fromStatus(status);
			category = Category.fromInteger(status);

		} catch (IllegalArgumentException oops) {
			category = Category.FatalException;
			type = Type.UnrecognizedError;
		}
	}

	public Type getType() {
		return type;
	}

	public Category getCategory() {
		return category;
	}

	public enum Type {
		GenericException(200),
		MissingArgument(201),
		InvalidArgument(202),
		IncompatibleParameters(203),
		UnconsistentData(204),
		ObjectNotFound(210),
		ObjectAlreadyExists(211),
		NoChange(212),
		NonUniqObject(213),
		TooManyChanges(214),
		ActionPending(220),
		FunctionDisabled(230),
		ActionInProgress(240),
		ActionImpossible(241),
		ActionAlreadyDone(242),
		NotImplemented(250),
		Deprecated(251),
		Unavailable(252),
		NoAnswer(266),
		UnknownParameter(299),

		// sessions
		SessionExpired(301),
		SessionNotFound(302),
		SessionCorrupted(303),
		TooManySessions(304),

		LoginFailed(310),
		IpSecurityError(320),

		// rights
		PermissionDenied(401),
		InsufficientRights(402),
		ReadonlySession(403),

		OfferUpgradeNeeded(410),
		OrderRequired(430),
		AssociationIncompatible(431),
		AgreementRequired(432),
		AccountTooLow(433),
		BreachOfContract(434),
		ServiceExpired(435),

		QuotaReached(451),
		SecurityWarning(460),
		Forbidden(461),
		IntegrityCheckFailure(499),

		// server transient errors
		InternalError(500),
		DataBaseError(501),
		ServerUnreachable(504),

		ConnectionClosed(510),
		ConnectionFailed(511),
		ConnectionTimeout(512),

		ProtocolError(520),
		MessageTimeout(521),

		Timeout(530),

		ServerError(560),

		CacheReadError(551),
		CacheWriteError(552),

		// security
		UnrecognizedFormat(600),
		InvalidCode(601),
		CodeInjection(602),

		// locks
		IncompatibleState(701),
		NotSupported(702),
		ObjectIncompatible(703),
		BlockingOperation(704),
		NoMoreData(705),
		UnableToLock(706),
		LockedByAnother(710),
		LockedByCustomer(711),
		OutOfSync(720),

		ServiceInMaintenance(750),

		// FALLBACK
		UnrecognizedError(999);

		private int status;

		Type(int status) {
			this.status = status;
		}

		public int getStatus() {
			return status;
		}

		public static Type fromStatus(int status) throws IllegalArgumentException {
			for (Type test : Type.values()) {
				if (test.status == status) {
					return test;
				}
			}
			throw new IllegalArgumentException("Type whose status is " + status + " not recognized");
		}

	}

	public enum Category {
		CallException(2),
		SessionException(3),
		RightException(4),
		RecoverableException(5),
		SecurityException(6),
		LockException(7),
		FatalException(9);

		private int id;

		Category(int id) {
			this.id = id;
		}

		public Integer toInteger() {
			return id;
		}

		public static Category fromInteger(Integer status) throws IllegalArgumentException {
			if (status != null) {
				int prefix = status / 100;
				for (Category test : Category.values()) {
					if (test.id == prefix) {
						return test;
					}
				}
			}
			throw new IllegalArgumentException("Catagory from status " + status + " not recognized");
		}
	}
}
