/**
 * Copyright 2012, OVH. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */
package com.ovh.ws.api.http;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ovh.ws.api.OvhWsException;
import com.ovh.ws.api.OvhWsException.Type;
import com.ovh.ws.api.base.Hex;

public class RequestSigner {

	private static final Logger LOG = LoggerFactory.getLogger(RequestSigner.class);

	private static final String ALGORITHM = "SHA-1";
	private static final char SEPARATOR = '+';

	private static final String NONCE_PARAM = "X-OVH-NONCE";
	private static final String SIGNATURE_PARAM = "X-OVH-SIGN";

	public Map<String, String> getSigningHeaders(String url, String requestParams, String token) throws OvhWsException {
		String nonce = generateNonce();
		String signature = getRequestSignature(url, requestParams, nonce, token);

		Map<String, String> headers = new HashMap<String, String>();
		headers.put(NONCE_PARAM, nonce);
		headers.put(SIGNATURE_PARAM, signature);

		return headers;
	}

	private String getRequestSignature(String url, String requestData, String nonce, String token)
	        throws OvhWsException {
		try {
			String dataToDigest = getDataToDigest(url, requestData, nonce, token);
			LOG.debug("Data used to sign : {}", dataToDigest);

			String signature = getSignature(dataToDigest);
			LOG.debug("Resulting signature : {}", signature);

			return signature;

		} catch (UnsupportedEncodingException e) {
			throw new OvhWsException(Type.UnrecognizedError, "Could not sign request", e);

		} catch (NoSuchAlgorithmException e) {
			throw new OvhWsException(Type.UnrecognizedError, "Could not sign request", e);
		}
	}

	private String getDataToDigest(String url, String requestData, String nonce, String token) {
		StringBuilder builder = new StringBuilder(url);
		builder.append(SEPARATOR).append(requestData);
		builder.append(SEPARATOR).append(nonce);
		if (token != null) {
			builder.append(SEPARATOR).append(token);
		}
		return builder.toString().replaceAll("\\s", "");
	}

	private String getSignature(String dataToDigest) throws NoSuchAlgorithmException, UnsupportedEncodingException {
		MessageDigest digest = MessageDigest.getInstance(ALGORITHM);
		byte[] hash = digest.digest(dataToDigest.getBytes("UTF-8"));
		return Hex.encode(hash);
	}

	private String generateNonce() {
		return System.currentTimeMillis() + "." + Math.random();
	}

}
