/*
 * Decompiled with CFR 0.152.
 */
package com.palantir.gradle.jdks.setup;

import com.palantir.gradle.jdks.setup.AliasContentCert;
import com.palantir.gradle.jdks.setup.ILogger;
import com.palantir.gradle.jdks.setup.common.CommandRunner;
import com.palantir.gradle.jdks.setup.common.CurrentOs;
import com.palantir.gradle.jdks.setup.common.Os;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

public final class CaResources {
    private static final BigInteger PALANTIR_3RD_GEN_SERIAL = new BigInteger("18126334688741185161");
    private static final String PALANTIR_3RD_GEN_CERTIFICATE = "Palantir3rdGenRootCa";
    private final ILogger logger;

    public CaResources(ILogger logger) {
        this.logger = logger;
    }

    public Optional<AliasContentCert> readPalantirRootCaFromSystemTruststore() {
        return this.systemCertificates().flatMap(CaResources::selectPalantirCertificate);
    }

    public void importAllSystemCerts(Path jdkInstallationDirectory) {
        this.systemCertificates().ifPresent(certs -> this.importCertificates(jdkInstallationDirectory, CaResources.parseCerts(certs)));
    }

    private void importCertificates(Path jdkInstallationDirectory, List<X509Certificate> certificates) {
        try {
            char[] passwd = "changeit".toCharArray();
            Path jksPath = jdkInstallationDirectory.resolve("lib/security/cacerts");
            KeyStore jks = this.loadKeystore(passwd, jksPath);
            Set<X509Certificate> existingCertificates = CaResources.getExistingCertificates(jks);
            List newCertificates = certificates.stream().filter(CaResources::isValid).filter(CaResources::isCertUsedForTls).filter(certificate -> !existingCertificates.contains(certificate)).collect(Collectors.toList());
            for (X509Certificate certificate2 : newCertificates) {
                String alias = this.getAlias(certificate2);
                this.logger.log(String.format("Certificate %s imported successfully into the JDK truststore from the system truststore.", alias));
                jks.setCertificateEntry(alias, certificate2);
            }
            jks.store(new BufferedOutputStream(new FileOutputStream(jksPath.toFile())), passwd);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new RuntimeException("Failed to import certificates", e);
        }
    }

    private static Set<X509Certificate> getExistingCertificates(KeyStore keyStore) {
        try {
            return Collections.list(keyStore.aliases()).stream().map(alias -> {
                try {
                    return Optional.ofNullable(keyStore.getCertificate((String)alias));
                }
                catch (KeyStoreException e) {
                    throw new RuntimeException("Failed to load keystore", e);
                }
            }).filter(Optional::isPresent).map(Optional::get).filter(certificate -> X509Certificate.class.isAssignableFrom(certificate.getClass())).map(X509Certificate.class::cast).collect(Collectors.toSet());
        }
        catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private static boolean isCertUsedForTls(X509Certificate certificate) {
        return CaResources.hasCaCertUsage(certificate) && CaResources.isTlsServerAuthentication(certificate);
    }

    private static boolean isValid(X509Certificate certificate) {
        try {
            certificate.checkValidity();
            return true;
        }
        catch (CertificateExpiredException | CertificateNotYetValidException e) {
            return false;
        }
    }

    private static boolean hasCaCertUsage(X509Certificate certificate) {
        boolean[] keyUsage = certificate.getKeyUsage();
        if (keyUsage == null) {
            return true;
        }
        if (keyUsage[0] && keyUsage[2]) {
            return true;
        }
        return keyUsage[5] && certificate.getBasicConstraints() != -1;
    }

    private static boolean isTlsServerAuthentication(X509Certificate certificate) {
        try {
            List<String> extendedKeyUsages = certificate.getExtendedKeyUsage();
            if (extendedKeyUsages == null) {
                return true;
            }
            return extendedKeyUsages.contains("1.3.6.1.5.5.7.3.1");
        }
        catch (CertificateParsingException e) {
            throw new RuntimeException(e);
        }
    }

    public String getAlias(X509Certificate certificate) {
        String distinguishedName = certificate.getIssuerX500Principal().getName();
        String serialNumber = certificate.getSerialNumber().toString();
        try {
            LdapName ldapName = new LdapName(distinguishedName);
            for (Rdn rdn : ldapName.getRdns()) {
                if (!"CN".equalsIgnoreCase(rdn.getType())) continue;
                return String.format("GradleJdks_%s_%s", ((String)rdn.getValue()).replaceAll("\\s", ""), serialNumber);
            }
        }
        catch (InvalidNameException e) {
            this.logger.logError(String.format("Failed to extract ldapName from %s", distinguishedName));
        }
        return String.format("GradleJdks_%s_%s", distinguishedName.replaceAll("\\s", ""), serialNumber);
    }

    private KeyStore loadKeystore(char[] password, Path location) {
        KeyStore keyStore;
        BufferedInputStream keystoreStream = new BufferedInputStream(Files.newInputStream(location, new OpenOption[0]));
        try {
            KeyStore keystore = KeyStore.getInstance("JKS");
            keystore.load(keystoreStream, password);
            keyStore = keystore;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)keystoreStream).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                this.logger.log(String.format("Couldn't load jks, an exception occurred %s", e));
                throw new RuntimeException(String.format("Couldn't load keystore %s", location), e);
            }
        }
        ((InputStream)keystoreStream).close();
        return keyStore;
    }

    private Optional<byte[]> systemCertificates() {
        Os os = CurrentOs.get();
        switch (os) {
            case MACOS: {
                return Optional.of(CaResources.macosSystemCertificates());
            }
            case LINUX_MUSL: 
            case LINUX_GLIBC: {
                return Optional.of(CaResources.linuxSystemCertificates());
            }
            case WINDOWS: {
                this.logger.logError(String.format("Not attempting to read Palantir CA from system truststore as OS type '%s' does not yet support this", os.uiName()));
                return Optional.empty();
            }
        }
        throw new IllegalStateException("Unreachable code; all Os enum values should be handled");
    }

    private static byte[] macosSystemCertificates() {
        return Stream.of("/Library/Keychains/System.keychain").map(x$0 -> Paths.get(x$0, new String[0])).filter(x$0 -> Files.exists(x$0, new LinkOption[0])).map(CaResources::macosSystemCertificates).collect(Collectors.joining("\n")).getBytes(StandardCharsets.UTF_8);
    }

    private static String macosSystemCertificates(Path keyChainPath) {
        return CommandRunner.runWithOutputCollection((ProcessBuilder)new ProcessBuilder(new String[0]).command("security", "export", "-t", "certs", "-f", "pemseq", "-k", keyChainPath.toAbsolutePath().toString()));
    }

    private static byte[] linuxSystemCertificates() {
        List<Path> possibleCaCertificatePaths = List.of(Paths.get("/etc/ssl/certs/ca-certificates.crt", new String[0]), Paths.get("/etc/ssl/certs/ca-bundle.crt", new String[0]));
        return possibleCaCertificatePaths.stream().filter(x$0 -> Files.exists(x$0, new LinkOption[0])).map(caCertificatePath -> {
            try {
                return Files.readString(caCertificatePath);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to read CA certs from " + caCertificatePath, e);
            }
        }).collect(Collectors.joining("\n")).getBytes(StandardCharsets.UTF_8);
    }

    private static Optional<AliasContentCert> selectPalantirCertificate(byte[] multipleCertificateBytes) {
        return CaResources.selectCertificates(multipleCertificateBytes, Map.of(PALANTIR_3RD_GEN_SERIAL.toString(), PALANTIR_3RD_GEN_CERTIFICATE)).findFirst();
    }

    private static Stream<AliasContentCert> selectCertificates(byte[] multipleCertificateBytes, Map<String, String> certSerialNumbersToAliases) {
        return CaResources.parseCerts(multipleCertificateBytes).stream().filter(cert -> certSerialNumbersToAliases.containsKey(cert.getSerialNumber().toString())).map(cert -> new AliasContentCert((String)certSerialNumbersToAliases.get(cert.getSerialNumber().toString()), CaResources.encodeCertificate(cert)));
    }

    static List<X509Certificate> parseCerts(byte[] multipleCertificateBytes) {
        CertificateFactory certificateFactory;
        try {
            certificateFactory = CertificateFactory.getInstance("X.509");
        }
        catch (CertificateException e) {
            throw new RuntimeException("Could not make X.509 certificate factory", e);
        }
        ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>();
        ByteArrayInputStream baos = new ByteArrayInputStream(multipleCertificateBytes);
        int i = 0;
        while (baos.available() != 0) {
            block5: {
                try {
                    certs.add((X509Certificate)certificateFactory.generateCertificate(baos));
                }
                catch (CertificateException e) {
                    if (e.getMessage().contains("Duplicate extensions not allowed") || e.getMessage().contains("no more data allowed for version 1 certificate")) break block5;
                    if (e.getMessage().contains("Empty input")) break;
                    throw new RuntimeException("Failed to parse cert " + i, e);
                }
            }
            ++i;
        }
        return Collections.unmodifiableList(certs);
    }

    private static String encodeCertificate(Certificate palantirCert) {
        Base64.Encoder encoder = Base64.getMimeEncoder(64, "\n".getBytes(StandardCharsets.UTF_8));
        try {
            return String.join((CharSequence)"\n", "-----BEGIN CERTIFICATE-----", encoder.encodeToString(palantirCert.getEncoded()), "-----END CERTIFICATE-----");
        }
        catch (CertificateEncodingException e) {
            throw new RuntimeException("Could not convert Palantir cert back to regular", e);
        }
    }
}

