/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi;
/*
 * This Class was created by Patrick J
 * on 25.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import android.content.Context;

import com.pddstudio.linodeapi.config.LinodePreferences;
import com.pddstudio.linodeapi.enums.ValidationType;
import com.pddstudio.linodeapi.interfaces.LinodeValidationCallback;
import com.pddstudio.linodeapi.internal.ApiAccessValidation;
import com.pddstudio.linodeapi.utils.Logger;

import java.util.LinkedList;
import java.util.List;

import io.paperdb.Paper;

/**
 * The base class to interact with Linode's service.
 * This class is a singleton instance. Once initialized it can be received everywhere as requested.
 */
public final class LinodeApi {

    private static final String LINODE_USERS = "linodeUserObj";

    private static LinodeApi linodeApi;

    private final Context context;
    private final LinodePreferences linodePreferences;
    private final LinodeValidationCallback linodeValidationCallback;

    private LinodeApi(Context context, LinodeValidationCallback linodeValidationCallback) {
        Paper.init(context);
        this.context = context;
        this.linodeValidationCallback = linodeValidationCallback;
        this.linodePreferences = LinodePreferences.getInstance(context);
        validateCredentials();
    }

    private void validateCredentials() {
        if(linodePreferences.isApiKeyAvailable() || (linodePreferences.isUsernameAvailable() && linodePreferences.isPasswordAvailable())) {
            new ApiAccessValidation(linodeValidationCallback, linodePreferences).validate();
        } else {
            Logger.log(this, "No API-Key / Credentials available. Skipping validation.");
            linodeValidationCallback.onValidationFinished(ValidationType.SKIPPED);
        }
    }

    /**
     * Method to initialize the LinodeApi singleton.
     * @param context - Any context object
     * @param linodeValidationCallback - The {@link LinodeValidationCallback} for the initialization process.
     */
    public static void init(Context context, LinodeValidationCallback linodeValidationCallback) {
        if(linodeApi == null) linodeApi = new LinodeApi(context, linodeValidationCallback);
    }

    /**
     * Static method to receive the {@link LinodeApi} singleton instance.
     * @return {@link LinodeApi} singleton
     *
     * @throws RuntimeException in case the instance isn't initialized
     */
    public static LinodeApi getInstance() {
        //forcing a crash in case the API isn't initialized
        if(linodeApi == null) throw new RuntimeException("Instance not found. Did you forget to initialize LinodeApi?");
        return linodeApi;
    }

    /**
     * Method to receive the {@link LinodePreferences} instance for this singleton.
     * @return {@link LinodePreferences}
     */
    public LinodePreferences getPreferences() {
        return linodePreferences;
    }

    /**
     * Method to receive the {@linkplain Context} object provided by initializing the {@link LinodeApi} singleton
     * @return {@linkplain Context}
     */
    public Context getContext() {
        return context;
    }

    /**
     * Method to request an API key using the user's credentials.
     * <p>This is an asynchronous action, callbacks will be delivered to the provided {@link com.pddstudio.linodeapi.LinodeLogin.Callback}</p>
     * @param loginCallback - Callback to receive updates from the asynchronous operation
     * @param username - The username for the Linode Account
     * @param password - The password for the Linode Account
     */
    public void requestApiKey(LinodeLogin.Callback loginCallback, String username, String password) {
        new LinodeLogin(username, password)
                .setCallback(loginCallback)
                .requestApiKey();
    }

    /**
     * Method to request an API key using the user's credentials and his token.
     * This is required in case the user has Two-Factor-Authentication activated.
     * <p>This is an asynchronous action, callbacks will be delivered to the provided {@link com.pddstudio.linodeapi.LinodeLogin.Callback}</p>
     * @param loginCallback - Callback to receive updates from the asynchronous operation
     * @param username - The username for the Linode Account
     * @param password - The password for the Linode Account
     */
    public void requestApiKey(LinodeLogin.Callback loginCallback, String username, String password, String token) {
        new LinodeLogin(username, password)
                .withToken(token)
                .setCallback(loginCallback)
                .requestApiKey();
    }

    /**
     * Create a new {@link com.pddstudio.linodeapi.LinodeRequest.Builder} instance.
     * This is required in order to make new API calls on the Linode API.
     * @return A new {@link com.pddstudio.linodeapi.LinodeRequest.Builder} instance
     */
    public LinodeRequest.Builder newRequest() {
        return new LinodeRequest.Builder(context);
    }

    /**
     * Method to persist the given {@link LinodeUser}.
     * This call is helpful in case you want to receive already logged in {@link LinodeUser} objects.
     * <p>This method should be used in case you want this library to persist and restore {@link LinodeUser} and manage the account credentials</p>
     * @param linodeUser - {@link LinodeUser} to persist.
     */
    public void saveLinodeUser(LinodeUser linodeUser) {
        LinkedList<LinodeUser> users = Paper.book().read(LINODE_USERS, new LinkedList<LinodeUser>());
        for(int i = 0; i < users.size(); i++) {
            LinodeUser user = users.get(i);
            if(user.getUsername().toLowerCase().equals(linodeUser.getUsername().toLowerCase())) users.remove(i);
        }
        users.add(linodeUser);
        Paper.book().write(LINODE_USERS, users);
    }

    /**
     * Method to receive all persistent {@link LinodeUser} objects.
     * @return A {@linkplain List} - Can be empty if no {@link LinodeUser} is saved.
     */
    public List<LinodeUser> getLinodeUsers() {
        return Paper.book().read(LINODE_USERS, new LinkedList<LinodeUser>());
    }

    /**
     * Method to switch between multiple {@link LinodeUser} instances during runtime.
     * @param linodeUser - The new {@link LinodeUser} which should be used for further API calls
     */
    public void switchAccount(LinodeUser linodeUser) {
        linodePreferences.setCurrentUsername(linodeUser.getUsername());
        linodePreferences.setCurrentPassword(linodeUser.getPassword());
        linodePreferences.setCurrentApiKey(linodeUser.getToken());
    }

}
