/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi;
/*
 * This Class was created by Patrick J
 * on 29.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import com.pddstudio.linodeapi.async.AsyncLoginTask;
import com.pddstudio.linodeapi.models.data.ErrorCodeData;

/**
 * This class is used for logging into an user's account to receive an API key.
 */
public class LinodeLogin {

    /**
     * The callback interface for the asynchronous login process.
     */
    public interface Callback {
        /**
         * Gets invoked once the user has logged in successfully and an API key was received
         * @param linodeUser - An {@link LinodeUser} object representing the user which is logged in.
         */
        void onLoginSuccess(LinodeUser linodeUser);

        /**
         * Gets invoked in case the login process failed.
         * <p>This can happen due to multiple reasons, like no network connection, wrong user data
         * or missing token for Two-Factor-Authentication.</p>
         * @param errorCodeData - An {@link ErrorCodeData} array which describes the occurred error.
         */
        void onLoginFailed(ErrorCodeData[] errorCodeData);
    }

    private final String username;
    private final String password;
    private String token;
    private Callback callback;

    protected LinodeLogin(String username, String password) {
        this.username = username;
        this.password = password;
    }

    protected LinodeLogin withToken(String token) {
        this.token = token;
        return this;
    }
    protected LinodeLogin setCallback(Callback callback) {
        this.callback = callback;
        return this;
    }

    private void validate() {
        if(username == null || username.isEmpty() || password == null || password.isEmpty() || callback == null) throw new RuntimeException("LinodeLogin is missing Username, Password (must not be null or empty) or Callback interface!");
    }

    /**
     * Starts the asynchronous request to receive an API key.
     */
    public void requestApiKey() {
        validate();
        AsyncLoginTask asyncLoginTask = new AsyncLoginTask(this);
        asyncLoginTask.execute();
    }

    /**
     * Returns the username used for this login request
     * @return
     */
    public String getUsername() {
        return username;
    }

    /**
     * Returns the password used for this login request
     * @return
     */
    public String getPassword() {
        return password;
    }

    /**
     * Returns whether a token was used for this login request.
     * <p>True = login with token
     * False = login without token</p>
     * @return
     */
    public boolean withToken() {
        return token != null;
    }

    /**
     * Returns the token used for this login request.
     * <p>In case the login request was send without a token this value will be null</p>
     * @return
     */
    public String getToken() {
        return token;
    }

    /**
     * Returns the callback used for this login request.
     * @return
     */
    public Callback getCallback() {
        return callback;
    }

}
