/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi;
/*
 * This Class was created by Patrick J
 * on 25.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import android.content.Context;

import com.pddstudio.linodeapi.async.AsyncApiRequest;
import com.pddstudio.linodeapi.interfaces.ApiAction;
import com.pddstudio.linodeapi.interfaces.LinodeRequestCallback;
import com.pddstudio.linodeapi.utils.UrlBuilder;

/**
 * Class to execute API calls on the Linode API.
 */
public class LinodeRequest {

    private final Context context;
    private final ApiAction apiAction;
    private final LinodeRequestCallback linodeRequestCallback;
    private final UrlBuilder urlBuilder;

    private LinodeRequest(Builder builder) {
        this.context = builder.context;
        this.apiAction = builder.apiAction;
        this.linodeRequestCallback = builder.linodeRequestCallback;
        this.urlBuilder = builder.urlBuilder;
    }

    /**
     * Executes the chained {@link Builder} call.
     * <p>This request is asynchronous. All callbacks will be delivered to the provided {@link LinodeRequestCallback}</p>
     */
    public void execute() {
        String apiUrl = urlBuilder.buildWithCustomParams();
        AsyncApiRequest asyncApiRequest = new AsyncApiRequest(linodeRequestCallback, apiUrl, apiAction.getBaseModel());
        asyncApiRequest.execute();
    }

    /**
     * Builder class to chain the desired API calls.
     */
    public static class Builder {

        private final Context context;
        private ApiAction apiAction;
        private LinodeRequestCallback linodeRequestCallback;
        private UrlBuilder urlBuilder;

        protected Builder(Context context) {
            this.context = context;
        }

        /**
         * Set the desired API action which should be executed.
         * @param apiAction
         * @return
         */
        public Builder setApiAction(ApiAction apiAction) {
            this.apiAction = apiAction;
            this.urlBuilder = new UrlBuilder(context, apiAction);
            return this;
        }

        /**
         * Add custom API params.
         * This might be used in case you want to add custom parameters to the API request.
         * <p>This call is hardly used. Make sure you know what restrictions your custom parameters will bring.
         * The API might not work as usual with custom parameters.</p>
         * @param action - Custom parameter action.
         * @param value - Custom parameter value for the given action.
         * @return
         */
        public Builder addCustomApiParams(String action, String value) {
            if(urlBuilder != null && action != null && !action.isEmpty() && value != null && !value.isEmpty()) {
                urlBuilder.appendCustom(action, value);
            }
            return this;
        }

        /**
         * Set the {@link LinodeRequestCallback} to receive callbacks.
         * <p>This must not be null or skipped!</p>
         * @param linodeRequestCallback
         * @return
         */
        public Builder setRequestCallback(LinodeRequestCallback linodeRequestCallback) {
            this.linodeRequestCallback = linodeRequestCallback;
            return this;
        }

        /**
         * Builds the {@link LinodeRequest} without executing it.
         * @return Generated {@link LinodeRequest} instance.
         */
        public LinodeRequest build() {
            return new LinodeRequest(this);
        }

        /**
         * Builds the {@link LinodeRequest} and executes it.
         */
        public void execute() {
            LinodeRequest linodeRequest = new LinodeRequest(this);
            linodeRequest.execute();
        }

    }

}
