/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi.actions;
/*
 * This Class was created by Patrick J
 * on 29.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import com.pddstudio.linodeapi.abstracts.BaseModel;
import com.pddstudio.linodeapi.interfaces.ApiAction;
import com.pddstudio.linodeapi.models.LinodeInfo;
import com.pddstudio.linodeapi.models.LinodeListInfo;

/**
 * This family of API methods allows you to manage your existing Linodes, or easily create and deploy new ones
 * @see <a href="https://www.linode.com/api/linode">https://www.linode.com/api/linode</a>
 */
public enum Linode implements ApiAction {

    /**
     * Issues a boot job for the provided ConfigID. If no ConfigID is provided boots the last used configuration profile, or the first configuration profile if this Linode has never been booted.
     * @see <a href="https://www.linode.com/api/linode/linode.boot">https://www.linode.com/api/linode/linode.boot</a>
     */
    BOOT("linode.boot", new LinodeInfo()),

    /**
     * Creates a new Linode, assigns you full privileges, and then clones the specified LinodeID to the new Linode. There is a limit of 5 active clone operations per source Linode. It is recommended that the source Linode be powered down during the clone.
     * @see <a href="https://www.linode.com/api/linode/linode.clone">https://www.linode.com/api/linode/linode.clone</a>
     */
    CLONE("linode.clone", new LinodeInfo()),

    /**
     * Creates a Linode and assigns you full privileges. There is a 75-linodes-per-hour limiter.
     * @see <a href="https://www.linode.com/api/linode/linode.create">https://www.linode.com/api/linode/linode.create</a>
     */
    CREATE("linode.create", new LinodeInfo()),

    /**
     * Immediately removes a Linode from your account and issues a pro-rated credit back to your account, if applicable. To prevent accidental deletes, this requires the Linode has no Disk images. You must first delete its disk images."
     * @see <a href="https://www.linode.com/api/linode/linode.delete">https://www.linode.com/api/linode/linode.delete</a>
     */
    DELETE("linode.delete", new LinodeInfo()),

    /**
     * Returns a list of all Linodes user has access or delete to, including some properties. Status values are -1: Being Created, 0: Brand New, 1: Running, and 2: Powered Off.
     * @see <a href="https://www.linode.com/api/linode/linode.list">https://www.linode.com/api/linode/linode.list</a>
     */
    LIST("linode.list", new LinodeListInfo()),

    /**
     * Issues a shutdown, and then boot job for a given LinodeID.
     * @see <a href="https://www.linode.com/api/linode/linode.reboot">https://www.linode.com/api/linode/linode.reboot</a>
     */
    REBOOT("linode.reboot", new LinodeInfo()),

    /**
     * Resizes a Linode from one plan to another. Immediately shuts the Linode down, charges/credits the account, and issue a migration to another host server.
     * @see <a href="https://www.linode.com/api/linode/linode.resize">https://www.linode.com/api/linode/linode.resize</a>
     */
    RESIZE("linode.resize", new LinodeInfo()),

    /**
     * Issues a shutdown job for a given LinodeID.
     * @see <a href="https://www.linode.com/api/linode/linode.shutdown">https://www.linode.com/api/linode/linode.shutdown</a>
     */
    SHUTDOWN("linode.shutdown", new LinodeInfo()),

    /**
     * Updates a Linode's properties.
     * @see <a href="https://www.linode.com/api/linode/linode.update">https://www.linode.com/api/linode/linode.update</a>
     */
    UPDATE("linode.update", new LinodeInfo());

    private final String apiCommand;
    private final BaseModel baseModelType;

    Linode(String apiCommand, BaseModel baseModelType) {
        this.apiCommand = apiCommand;
        this.baseModelType = baseModelType;
    }

    @Override
    public String getApiCommand() {
        return apiCommand;
    }

    @Override
    public BaseModel getBaseModel() {
        return baseModelType;
    }


    public enum Config implements ApiAction {

        /**
         * Creates a Linode Configuration Profile.
         * @see <a href="https://www.linode.com/api/linode/linode.config.create">https://www.linode.com/api/linode/linode.config.create</a>
         */
        CREATE("linode.config.create", new LinodeInfo()),

        /**
         * Deletes a Linode Configuration Profile.
         * @see <a href="https://www.linode.com/api/linode/linode.config.delete">https://www.linode.com/api/linode/linode.config.delete</a>
         */
        DELETE("linode.config.delete", new LinodeInfo()),

        /**
         * Lists a Linode's Configuration Profiles.
         * @see <a href="https://www.linode.com/api/linode/linode.config.list">https://www.linode.com/api/linode/linode.config.list</a>
         */
        LIST("linode.config.list", new LinodeListInfo()),

        /**
         * Updates a Linode Configuration Profile.
         * @see <a href="https://www.linode.com/api/linode/linode.config.update">https://www.linode.com/api/linode/linode.config.update</a>
         */
        UPDATE("linode.config.update", new LinodeInfo());

        private final String apiCommand;
        private final BaseModel baseModelType;

        Config(String apiCommand, BaseModel baseModelType) {
            this.apiCommand = apiCommand;
            this.baseModelType = baseModelType;
        }

        @Override
        public String getApiCommand() {
            return apiCommand;
        }

        @Override
        public BaseModel getBaseModel() {
            return baseModelType;
        }

    }

    public enum Disk implements ApiAction {

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.create">https://www.linode.com/api/linode/linode.disk.create</a>
         */
        CREATE("linode.disk.create", new LinodeInfo()),

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.createfromdistribution">https://www.linode.com/api/linode/linode.disk.createfromdistribution</a>
         */
        CREATE_FROM_DISTRIBUTION("linode.disk.createfromdistribution", new LinodeInfo()),

        /**
         * Creates a new disk from a previously imagized disk.
         * @see <a href="https://www.linode.com/api/linode/linode.disk.createfromimage">https://www.linode.com/api/linode/linode.disk.createfromimage</a>
         */
        CREATE_FROM_IMAGE("linode.disk.createfromimage", new LinodeInfo()),

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.createfromstackscript">https://www.linode.com/api/linode/linode.disk.createfromstackscript</a>
         */
        CREATE_FROM_STACK_SCRIPT("linode.disk.createfromstackscript", new LinodeInfo()),

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.delete">https://www.linode.com/api/linode/linode.disk.delete</a>
         */
        DELETE("linode.disk.delete", new LinodeInfo()),

        /**
         * Performs a bit-for-bit copy of a disk image.
         * @see <a href="https://www.linode.com/api/linode/linode.disk.duplicate">https://www.linode.com/api/linode/linode.disk.duplicate</a>
         */
        DUPLICATE("linode.disk.duplicate", new LinodeInfo()),

        /**
         * Creates a gold-master image for future deployments
         * @see <a href="https://www.linode.com/api/linode/linode.disk.imagize">https://www.linode.com/api/linode/linode.disk.imagize</a>
         */
        IMAGIZE("linode.disk.imagize", new LinodeInfo()),

        /**
         * Status values are 1: Ready and 2: Being Deleted.
         * @see <a href="https://www.linode.com/api/linode/linode.disk.list">https://www.linode.com/api/linode/linode.disk.list</a>
         */
        LIST("linode.disk.list", new LinodeListInfo()),

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.resize">https://www.linode.com/api/linode/linode.disk.resize</a>
         */
        RESIZE("linode.disk.resize", new LinodeInfo()),

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.disk.update">https://www.linode.com/api/linode/linode.disk.update</a>
         */
        UPDATE("linode.disk.update", new LinodeInfo());

        private final String apiCommand;
        private final BaseModel baseModelType;

        Disk(String apiCommand, BaseModel baseModelType) {
            this.apiCommand = apiCommand;
            this.baseModelType = baseModelType;
        }

        @Override
        public String getApiCommand() {
            return apiCommand;
        }

        @Override
        public BaseModel getBaseModel() {
            return baseModelType;
        }

    }

    public enum IP implements ApiAction {

        /**
         * Assigns a Private IP to a Linode. Returns the IPAddressID that was added.
         * @see <a href="https://www.linode.com/api/linode/linode.ip.addprivate">https://www.linode.com/api/linode/linode.ip.addprivate</a>
         */
        ADD_PRIVATE("linode.ip.addprivate", new LinodeInfo()),

        /**
         * Assigns a Public IP to a Linode. Returns the IPAddressID and IPAddress that was added.
         * @see <a href="https://www.linode.com/api/linode/linode.ip.addpublic">https://www.linode.com/api/linode/linode.ip.addpublic</a>
         */
        ADD_PUBLIC("linode.ip.addpublic", new LinodeInfo()),

        /**
         * Returns the IP addresses of all Linodes you have access to.
         * @see <a href="https://www.linode.com/api/linode/linode.ip.list">https://www.linode.com/api/linode/linode.ip.list</a>
         */
        LIST("linode.ip.list", new LinodeListInfo()),

        /**
         * Sets the rDNS name of a Public IP. Returns the IPAddressID and IPAddress that were updated.
         * @see <a href="https://www.linode.com/api/linode/linode.ip.setrdns">https://www.linode.com/api/linode/linode.ip.setrdns</a>
         */
        SET_RDNS("linode.ip.setrdns", new LinodeInfo()),

        /**
         * Exchanges Public IP addresses between two Linodes within a Datacenter. The destination of the IP Address can be designated by either the toLinodeID or withIPAddressID parameter. Returns the resulting relationship of the Linode and IP Address parameters. When performing a one directional swap, the source is represented by the first of the two resultant array members.
         * @see <a href="https://www.linode.com/api/linode/linode.ip.swap">https://www.linode.com/api/linode/linode.ip.swap</a>
         */
        SWAP("linode.ip.swap", new LinodeListInfo());

        private final String apiCommand;
        private final BaseModel baseModelType;

        IP(String apiCommand, BaseModel baseModelType) {
            this.apiCommand = apiCommand;
            this.baseModelType = baseModelType;
        }

        @Override
        public String getApiCommand() {
            return apiCommand;
        }

        @Override
        public BaseModel getBaseModel() {
            return baseModelType;
        }
    }

    public enum Job implements ApiAction {

        /**
         * @see <a href="https://www.linode.com/api/linode/linode.job.list">https://www.linode.com/api/linode/linode.job.list</a>
         */
        LIST("linode.job.list", new LinodeListInfo());

        private final String apiCommand;
        private final BaseModel baseModelType;

        Job(String apiCommand, BaseModel baseModelType) {
            this.apiCommand = apiCommand;
            this.baseModelType = baseModelType;
        }

        @Override
        public String getApiCommand() {
            return apiCommand;
        }

        @Override
        public BaseModel getBaseModel() {
            return baseModelType;
        }
    }

}