/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi.async;
/*
 * This Class was created by Patrick J
 * on 28.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import android.os.AsyncTask;

import com.google.gson.Gson;
import com.pddstudio.linodeapi.LinodeResponse;
import com.pddstudio.linodeapi.abstracts.BaseModel;
import com.pddstudio.linodeapi.interfaces.LinodeRequestCallback;
import com.pddstudio.linodeapi.utils.Logger;
import com.pddstudio.linodeapi.utils.TypeUtils;

import java.io.IOException;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;

public class AsyncApiRequest extends AsyncTask<Void, Void, Object> {

    private final LinodeRequestCallback linodeRequestCallback;
    private final String apiCallUrl;
    private final BaseModel baseModel;

    public AsyncApiRequest(LinodeRequestCallback linodeRequestCallback, String apiCallUrl, BaseModel baseModel) {
        this.linodeRequestCallback = linodeRequestCallback;
        this.apiCallUrl = apiCallUrl;
        this.baseModel = baseModel;
    }

    @Override
    public void onPreExecute() {
        linodeRequestCallback.onRequestStarted();
    }

    @Override
    protected Object doInBackground(Void... params) {

        //create the necessary components for deserialization
        OkHttpClient okHttpClient = new OkHttpClient();
        Gson gson = new Gson();
        Logger.log(this, "Building and executing request for given URL: " + apiCallUrl);

        try {

            Request request = new Request.Builder().url(apiCallUrl).build();
            Response response = okHttpClient.newCall(request).execute();
            if(!response.isSuccessful()) this.cancel(true);
            Object responseData = gson.fromJson(response.body().charStream(), baseModel.getModelClass());
            if(responseData != null) return responseData;

        } catch (IOException io) {
            Logger.log(this, "API call execution failed! Reason:\n" + io.getMessage());
            this.cancel(true);
        }

        return null;
    }

    @Override
    protected void onCancelled() {
        Logger.log(this, "AsyncApiRequest cancelled. This might happen by an exception or in case the response wasn't parseable.");
        linodeRequestCallback.onRequestFailed(null);
    }

    @Override
    public void onPostExecute(Object responseData) {
        boolean dataExist = responseData != null;
        Logger.log(this, "API Request executed. Checking if response object is filled with data: " + dataExist);
        LinodeResponse<Object> linodeResponse = new LinodeResponse<>(baseModel.getModelClass());
        linodeResponse.setResponse(TypeUtils.convertType(responseData, baseModel.getClass()));
        linodeRequestCallback.onDataReceived(linodeResponse);
    }

}
