/*
 *
 *     Copyright (c) 2016 Patrick J
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */

package com.pddstudio.linodeapi.internal;
/*
 * This Class was created by Patrick J
 * on 29.02.16. For more Details and licensing information
 * have a look at the README.md
 */

import android.os.AsyncTask;

import com.google.gson.Gson;
import com.pddstudio.linodeapi.config.Config;
import com.pddstudio.linodeapi.config.LinodePreferences;
import com.pddstudio.linodeapi.enums.ValidationType;
import com.pddstudio.linodeapi.interfaces.LinodeValidationCallback;
import com.pddstudio.linodeapi.models.TestEcho;
import com.pddstudio.linodeapi.models.UserInfo;
import com.pddstudio.linodeapi.models.data.ErrorCodeData;
import com.pddstudio.linodeapi.utils.Logger;
import com.pddstudio.linodeapi.utils.UrlUtils;

import java.io.IOException;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;

public class ApiAccessValidation {

    private final LinodeValidationCallback linodeValidationCallback;
    private final LinodePreferences linodePreferences;

    public ApiAccessValidation(LinodeValidationCallback linodeValidationCallback, LinodePreferences linodePreferences) {
        this.linodeValidationCallback = linodeValidationCallback;
        this.linodePreferences = linodePreferences;
    }

    public void validate() {
        new ValidationTask().execute();
    }


    private class ValidationTask extends AsyncTask<Void, Void, ValidationType> {

        ValidationTask() {}

        @Override
        public void onPreExecute() {
            linodeValidationCallback.onValidationStart();
        }

        @Override
        protected ValidationType doInBackground(Void... params) {
            Gson gson = new Gson();
            OkHttpClient okHttpClient = new OkHttpClient();

            String apiCallUrl;

            try {
                if(linodePreferences.isApiKeyAvailable()) {
                    //in case we already have an API key stored, we validate the key

                    apiCallUrl = UrlUtils.getEchoUrl(linodePreferences.getCurrentApiKey());

                    Logger.log(this, "Building and executing request for given URL: " + apiCallUrl);

                    Request request = new Request.Builder().url(apiCallUrl).build();
                    Response response = okHttpClient.newCall(request).execute();
                    if(!response.isSuccessful()) this.cancel(true);
                    TestEcho responseData = gson.fromJson(response.body().charStream(), TestEcho.class);

                    //actually the data should never be null
                    if(responseData != null) {
                        //checking whether the response key is the same as the one we send
                        if(responseData.getResponseData() != null && responseData.getResponseData().getApiEcho() != null) {
                            if(responseData.getResponseData().getApiEcho().toLowerCase().equals(Config.API_ECHO_VALUE.toLowerCase())) return ValidationType.SUCCESS;
                        } else if(responseData.getErrorCodeArray().length > 0) {
                            //loop through the error array and catch the authentication failed one.
                            for(ErrorCodeData errorCodeData : responseData.getErrorCodeArray()) {
                                if(errorCodeData.getErrorCode() == 4) {
                                    Logger.log(this, "ApiAccessValidation: " + errorCodeData.getErrorMessage() + " [Trying to request a new API Key now]");
                                    //try to request a new API key
                                    return validateCredentials();
                                }
                            }
                        } else {
                            return ValidationType.UNKNOWN_ERROR;
                        }
                    }

                } else {
                    return validateCredentials();
                }

            } catch (IOException io) {
                Logger.log(this, "API call execution failed! Reason:\n" + io.getMessage());
                this.cancel(true);
            }

            return null;
        }

        private ValidationType validateCredentials() throws IOException {
            if(linodePreferences.isUsernameAvailable() && linodePreferences.isPasswordAvailable()) {

                Gson gson = new Gson();
                OkHttpClient okHttpClient = new OkHttpClient();
                String requestUrl = "https://api.linode.com/?api_action=user.getapikey&username=" + linodePreferences.getCurrentUsername() + "&password=" + linodePreferences.getCurrentPassword() + "&expires=0&label=LINODE_ANDROID_API_KEY";

                Request request = new Request.Builder().url(requestUrl).build();
                Response response = okHttpClient.newCall(request).execute();
                if(!response.isSuccessful()) this.cancel(true);
                UserInfo userInfo = gson.fromJson(response.body().charStream(), UserInfo.class);

                //actually it should never be null
                if(userInfo != null) {
                    if(userInfo.getErrorCodeArray().length > 0) {
                        //in case we have an error return the proper ValidationType
                        for(ErrorCodeData errorCodeData : userInfo.getErrorCodeArray()) {
                            Logger.log(this, "Error Code : " + errorCodeData.getErrorCode() + " Error Message: " + errorCodeData.getErrorMessage());
                            if(errorCodeData.getErrorCode() == 44) return ValidationType.MISSING_TOKEN;
                        }
                        //TODO: replace with proper return type
                        return ValidationType.UNKNOWN_ERROR;
                    } else {
                        //in case the request was success, save the new API key and return SUCCESS
                        String apiKey = userInfo.getResponseData().getApiKey();
                        Logger.log(this, "API-Key received successfully [" + apiKey + "]");
                        linodePreferences.setCurrentApiKey(apiKey);
                        return ValidationType.SUCCESS;
                    }

                } else {
                    return ValidationType.UNKNOWN_ERROR;
                }

            } else {
                //in case we haven't saved any credentials yet
                return ValidationType.MISSING_CREDENTIALS;
            }
        }

        @Override
        protected void onCancelled() {
            linodeValidationCallback.onValidationFailed();
        }

        @Override
        public void onPostExecute(ValidationType validationType) {
            Logger.log(this, "Task finished. Returning Type: " + validationType.name());
            linodeValidationCallback.onValidationFinished(validationType);
        }

    }

}
