/****************************************************************************
 *
 * File:            MapCommon.java
 *
 * Description:     PDFTOOLS internal classes of map implementations.
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools;

import java.util.AbstractCollection;
import java.util.AbstractSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @hidden
 * In Java, the file name should be always the same as a class name.
 * This dummy class is to fulfill that requirement because some tools, e.g. javadoc, cannot process the file otherwise.
 */
class MapCommon
{}

class KeySet<K,V> extends AbstractSet<K>
{
    public KeySet(Map<K,V> map)
    {
        this.map = map;
    }

    @Override
    public boolean remove(Object o)
    {
        return map.remove(o) != null;
    }

    @Override
    public void clear()
    {
        map.clear();
    }

    @Override
    public boolean contains(Object object)
    {
        return map.containsKey(object);
    }

    @Override
    public Iterator<K> iterator()
    {
        return new KeyIterator<K,V>(map.entrySet().iterator());
    }

    @Override
    public int size()
    {
        return map.size();
    }

    private Map<K,V> map;
}

class ValueCollection<K,V> extends AbstractCollection<V>
{
    public ValueCollection(Map<K,V> map)
    {
        this.map = map;
    }

    @Override
    public Iterator<V> iterator()
    {
        return new ValueIterator<K,V>(map.entrySet().iterator());
    }

    @Override
    public int size()
    {
        return map.size();
    }

    private Map<K,V> map;
}

abstract class ProxyIterator<V,P> implements Iterator<V>
{
    public ProxyIterator(Iterator<P> it)
    {
        this.it = it;
    }

    @Override
    public boolean hasNext() {
        return it.hasNext();
    }

    @Override
    public void remove()
    {
        it.remove();
    }

    Iterator<P> it;
}

class KeyIterator<K,V> extends ProxyIterator<K,Entry<K,V>>
{
    public KeyIterator(Iterator<Entry<K, V>> it)
    {
        super(it);
    }

    @Override
    public K next()
    {
        return it.next().getKey();
    }
}

class ValueIterator<K,V> extends ProxyIterator<V,Entry<K,V>>
{

    public ValueIterator(Iterator<Entry<K, V>> it)
    {
        super(it);
    }

    @Override
    public V next()
    {
        return it.next().getValue();
    }
}
