/****************************************************************************
 *
 * File:            SignaturePaddingType.java
 *
 * Description:     SignaturePaddingType Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto;

/**
 * <h1>Padding scheme of the cryptographic signature algorithm</h1>
 * The signature algorithm is defined by the signing certificate's key type.
 * For example, RSA or ECDSA.
 * For some keys, e.g. RSA keys, there are different padding algorithms.
 * Some cryptographic providers let you set this padding algorithm.
 * However, this only has an effect on signatures created by the cryptographic provider itself.
 * All signed data acquired from external sources may use other signing algorithms;
 * more specifically, the issuer certificates of the trust chain, the time-stamp’s signature, 
 * or those used for the revocation information (CRL, OCSP).
 * It is recommended to verify that the algorithms of all signatures provide a similar level of security.
 */
public enum SignaturePaddingType  
{
    /**
     * <h1>Default padding scheme</h1>
     * The default padding scheme.
     * Used for the {@link SignatureAlgorithm#ECDSA } signature algorithm.
     */
    DEFAULT(0),
    /**
     * <h1>RSA with PKCS#1 padding scheme</h1>
     * Padding scheme for RSA keys that corresponds to the {@link SignatureAlgorithm#RSA_RSA } signature algorithm.
     */
    RSA_RSA(1),
    /**
     * <h1>RSA with Probabilistic Signature Scheme (PSS)</h1>
     * Padding scheme for RSA keys that corresponds to the {@link SignatureAlgorithm#RSA_SSA_PSS } signature algorithm.
     */
    RSA_SSA_PSS(2);

    SignaturePaddingType(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public static SignaturePaddingType fromValue(int value) 
    {
        switch (value) 
        {
            case 0: return DEFAULT;
            case 1: return RSA_RSA;
            case 2: return RSA_SSA_PSS;
        }
        throw new IllegalArgumentException("Unknown value for SignaturePaddingType: " + value);
    }

    /**
     * @hidden
     */
    public int getValue() 
    {
        return value;
    }

    private int value;
}
