/****************************************************************************
 *
 * File:            Certificate.java
 *
 * Description:     PDFTOOLS Certificate Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>A X.509 certificate</h1>
 */
public class Certificate extends NativeObject 
{
    protected Certificate(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Certificate createDynamicObject(long handle)
    {
        return new Certificate(handle);
    }




     /**
     * <h1>The name (subject) of the certificate (Getter)</h1>
     * The common name (CN) of the person or authority who owns the certificate.
     */
    public String getName()
    {
        String retVal = getNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The subject of the certificate (Getter)</h1>
     * The distinguished name (DN) of the person or authority who owns the certificate.
     * Formatted according to RFC 4514.
     */
    public String getSubject()
    {
        String retVal = getSubjectNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The name of the certificate's issuer (CA) (Getter)</h1>
     * The common name (CN) of the certificate authority (CA) who issued the certificate.
     */
    public String getIssuer()
    {
        String retVal = getIssuerNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The certificate's fingerprint (Getter)</h1>
     * The hex string representation of the certificate’s SHA-1 digest.
     */
    public String getFingerprint()
    {
        String retVal = getFingerprintNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>Whether the cryptographic provider has a private key for the certificate. (Getter)</h1>
     * Note that whether the private key is found and whether it can actually be used for signing may depend on the provider's login state.
     */
    public boolean getHasPrivateKey()
    {
        boolean retVal = getHasPrivateKeyNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }




    private native String getNameNative(long handle);
    private native String getSubjectNative(long handle);
    private native String getIssuerNative(long handle);
    private native String getFingerprintNative(long handle);
    private native boolean getHasPrivateKeyNative(long handle);

}

