/****************************************************************************
 *
 * File:            Provider.java
 *
 * Description:     PDFTOOLS Provider Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>Base class for cryptographic providers</h1>
 * <p>
 * The cryptographic provider manages certificates, their private keys and implements cryptographic algorithms.</p>
 * <p>
 * This SDK supports various different cryptographic providers.
 * The following list shows the signing certificate type that can be used for each provider.</p>
 * <p>
 * <ul>
 * <li>
 * <p>
 * <em>Soft Certificate</em>:</p>
 * <p>
 * Soft certificates are typically PKCS#12 files that have the extension {@code .pfx} or {@code .p12} and contain
 * the signing certificate as well as the private key and trust chain (issuer certificates).
 * Soft certificates can be used with the {@link com.pdftools.crypto.providers.builtin.Provider builtin.Provider}, where they can be loaded using
 * {@link com.pdftools.crypto.providers.builtin.Provider#createSignatureFromCertificate builtin.Provider.createSignatureFromCertificate}.
 * </p></li>
 * <li>
 * <p>
 * <em>Hardware Security Module (HSM)</em>:</p>
 * <p>
 * HSMs always offer very good PKCS#11 support, so the {@link com.pdftools.crypto.providers.pkcs11.Session pkcs11.Session} is suitable.
 * For more information and installation instructions, consult the separate document "TechNotePKCS11.pdf".
 * </p></li>
 * <li>
 * <p>
 * <em>USB Token or Smart Card</em>:</p>
 * <p>
 * These devices typically offer a PKCS#11 interface, so the recommended provider is the {@link com.pdftools.crypto.providers.pkcs11.Session pkcs11.Session}.
 * Note that in any case, signing documents is only possible in an interactive user session.
 * So these devices cannot be used in a service or web application environment.
 * </p></li>
 * <li>
 * <p>
 * <em>Swisscom Signing Service</em>:</p>
 * <p>
 * The {@link com.pdftools.crypto.providers.swisscomsigsrv.Session swisscomsigsrv.Session} supports both static and on-demand signing certificates.
 * </p></li>
 * <li>
 * <p>
 * <em>GlobalSign Digital Signing Service</em>:</p>
 * <p>
 * The {@link com.pdftools.crypto.providers.globalsigndss.Session globalsigndss.Session} supports all features of the service.
 * </p></li>
 * </ul></p>
 */
public abstract class Provider extends NativeObject implements AutoCloseable 
{
    protected Provider(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Provider createDynamicObject(long handle)
    {
        int type = getType(handle);
        switch (type)
        {
            case 1:
                return com.pdftools.crypto.providers.globalsigndss.Session.createDynamicObject(handle);
            case 2:
                return com.pdftools.crypto.providers.swisscomsigsrv.Session.createDynamicObject(handle);
            case 3:
                return com.pdftools.crypto.providers.pkcs11.Session.createDynamicObject(handle);
            case 4:
                return com.pdftools.crypto.providers.builtin.Provider.createDynamicObject(handle);
            default:
                return null;
        }
    }





    private static native int getType(long handle);
    /**
     * <h1>Close the object</h1>
     * Release all resources associated with the object.
     * @throws com.pdftools.PdfToolsException only explicitly stated in a superclass
     */
    public void close() 
        throws com.pdftools.PdfToolsException, 
               java.io.IOException
    {
        try
        {
            if (!close(getHandle()))
                throwLastError();
        }
        finally
        {
            setHandle(0);
        }
    }

    private native boolean close(long hObject);
}

