/****************************************************************************
 *
 * File:            SignatureConfiguration.java
 *
 * Description:     PDFTOOLS SignatureConfiguration Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers.builtin;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The signature configuration</h1>
 */
public class SignatureConfiguration extends com.pdftools.sign.SignatureConfiguration 
{
    protected SignatureConfiguration(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static SignatureConfiguration createDynamicObject(long handle)
    {
        return new SignatureConfiguration(handle);
    }




     /**
     * <h1>The message digest algorithm (Getter)</h1>
     * <p>
     * The algorithm used to hash the document and from which the cryptographic signature is created.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.HashAlgorithm#SHA256 pdftools.crypto.HashAlgorithm.SHA256}</p>
     */
    public com.pdftools.crypto.HashAlgorithm getHashAlgorithm()
    {
        int retVal = getHashAlgorithmNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.HashAlgorithm.fromValue(retVal);
    }

     /**
     * <h1>The message digest algorithm (Setter)</h1>
     * <p>
     * The algorithm used to hash the document and from which the cryptographic signature is created.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.HashAlgorithm#SHA256 pdftools.crypto.HashAlgorithm.SHA256}</p>
     *
     * @throws IllegalArgumentException If the value is invalid or not supported.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setHashAlgorithm(com.pdftools.crypto.HashAlgorithm value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setHashAlgorithmNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The padding type of the cryptographic signature (Getter)</h1>
     * Default: {@link com.pdftools.crypto.SignaturePaddingType#RSA_SSA_PSS pdftools.crypto.SignaturePaddingType.RSA_SSA_PSS} for RSA and for {@link com.pdftools.crypto.SignaturePaddingType#DEFAULT pdftools.crypto.SignaturePaddingType.DEFAULT} ECDSA certificates.
     */
    public com.pdftools.crypto.SignaturePaddingType getSignaturePaddingType()
    {
        int retVal = getSignaturePaddingTypeNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.SignaturePaddingType.fromValue(retVal);
    }

     /**
     * <h1>The padding type of the cryptographic signature (Setter)</h1>
     * Default: {@link com.pdftools.crypto.SignaturePaddingType#RSA_SSA_PSS pdftools.crypto.SignaturePaddingType.RSA_SSA_PSS} for RSA and for {@link com.pdftools.crypto.SignaturePaddingType#DEFAULT pdftools.crypto.SignaturePaddingType.DEFAULT} ECDSA certificates.
     *
     * @throws IllegalArgumentException If the value is invalid or not supported.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setSignaturePaddingType(com.pdftools.crypto.SignaturePaddingType value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setSignaturePaddingTypeNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The format (encoding) of the cryptographic signature (Getter)</h1>
     * Default: {@link com.pdftools.crypto.SignatureFormat#ETSI_CADES_DETACHED pdftools.crypto.SignatureFormat.ETSI_CADES_DETACHED}
     */
    public com.pdftools.crypto.SignatureFormat getSignatureFormat()
    {
        int retVal = getSignatureFormatNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.SignatureFormat.fromValue(retVal);
    }

     /**
     * <h1>The format (encoding) of the cryptographic signature (Setter)</h1>
     * Default: {@link com.pdftools.crypto.SignatureFormat#ETSI_CADES_DETACHED pdftools.crypto.SignatureFormat.ETSI_CADES_DETACHED}
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setSignatureFormat(com.pdftools.crypto.SignatureFormat value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setSignatureFormatNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to add a trusted time-stamp to the signature (Getter)</h1>
     * <p>
     * If {@code true} the {@link Provider#getTimestampUrl } must be set.</p>
     * <p>
     * Default: {@code false}</p>
     */
    public boolean getAddTimestamp()
    {
        boolean retVal = getAddTimestampNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to add a trusted time-stamp to the signature (Setter)</h1>
     * <p>
     * If {@code true} the {@link Provider#getTimestampUrl } must be set.</p>
     * <p>
     * Default: {@code false}</p>
     */
    public void setAddTimestamp(boolean value)
    {
        boolean retVal = setAddTimestampNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to add validation information (LTV) (Getter)</h1>
     * <p>
     * For signing certificates that do not offer validation (revocation) information (OCSP or CRL),
     * this property is ignored.</p>
     * <p>
     * If downloading validation information fails, an error {@link com.pdftools.NotFoundException } or {@link com.pdftools.HttpException } is generated.
     * See {@link com.pdftools.sign.WarningCategory#ADD_VALIDATION_INFORMATION_FAILED pdftools.sign.WarningCategory.ADD_VALIDATION_INFORMATION_FAILED} for a description of possible error causes
     * and solutions.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.ValidationInformation#EMBED_IN_DOCUMENT pdftools.crypto.ValidationInformation.EMBED_IN_DOCUMENT} if the signing certificate offers validation information and {@link com.pdftools.crypto.ValidationInformation#NONE pdftools.crypto.ValidationInformation.NONE} otherwise
     * </p>
     */
    public com.pdftools.crypto.ValidationInformation getValidationInformation()
    {
        int retVal = getValidationInformationNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.ValidationInformation.fromValue(retVal);
    }

     /**
     * <h1>Whether to add validation information (LTV) (Setter)</h1>
     * <p>
     * For signing certificates that do not offer validation (revocation) information (OCSP or CRL),
     * this property is ignored.</p>
     * <p>
     * If downloading validation information fails, an error {@link com.pdftools.NotFoundException } or {@link com.pdftools.HttpException } is generated.
     * See {@link com.pdftools.sign.WarningCategory#ADD_VALIDATION_INFORMATION_FAILED pdftools.sign.WarningCategory.ADD_VALIDATION_INFORMATION_FAILED} for a description of possible error causes
     * and solutions.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.ValidationInformation#EMBED_IN_DOCUMENT pdftools.crypto.ValidationInformation.EMBED_IN_DOCUMENT} if the signing certificate offers validation information and {@link com.pdftools.crypto.ValidationInformation#NONE pdftools.crypto.ValidationInformation.NONE} otherwise
     * </p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setValidationInformation(com.pdftools.crypto.ValidationInformation value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setValidationInformationNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native int getHashAlgorithmNative(long handle);
    private native boolean setHashAlgorithmNative(long handle, int value);
    private native int getSignaturePaddingTypeNative(long handle);
    private native boolean setSignaturePaddingTypeNative(long handle, int value);
    private native int getSignatureFormatNative(long handle);
    private native boolean setSignatureFormatNative(long handle, int value);
    private native boolean getAddTimestampNative(long handle);
    private native boolean setAddTimestampNative(long handle, boolean value);
    private native int getValidationInformationNative(long handle);
    private native boolean setValidationInformationNative(long handle, int value);

}

