/****************************************************************************
 *
 * File:            Module.java
 *
 * Description:     PDFTOOLS Module Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers.pkcs11;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The PKCS#11 driver module</h1>
 * <p>
 * The PKCS#11 driver module (middleware) manages the cryptographic devices of a particular type.</p>
 * <p>
 * <em>Note:</em> The PKCS#11 interface requires special handling of the driver modules:
 * <ul>
 * <li>
 * In each application, the module can only be loaded once,
 * so there can only be a single {@code Module} instance for each driver.
 * Since this object is fully thread-safe, it might be used by multiple threads though.
 * </li>
 * <li>
 * The object must be closed before the application terminates.
 * </li>
 * </ul></p>
 */
public class Module extends NativeObject implements AutoCloseable 
{
    protected Module(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Module createDynamicObject(long handle)
    {
        return new Module(handle);
    }



    /**
     * <h1>Load a PKCS#11 driver module</h1>
     * @param library 
     * <p>
     * The name or path to the driver module (middleware).
     * This can be found in the documentation of your cryptographic device.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * For Securosys SA Primus HSM or CloudsHSM use {@code primusP11.dll} on Windows and {@code libprimusP11.so}
     * on Linux.</li>
     * <li>
     * For Google Cloud HSM (Cloud KMS) use {@code libkmsp11.so} and {@link Session#createSignatureFromKeyLabel }</li>
     * <li>
     * For SafeNet Luna HSM use {@code cryptoki.dll} on Windows or {@code libCryptoki2_64.so} on Linux/UNIX.</li>
     * <li>
     * The CardOS API from Atos (Siemens) uses {@code siecap11.dll}</li>
     * <li>
     * The IBM 4758 cryptographic coprocessor uses {@code cryptoki.dll}</li>
     * <li>
     * Devices from Aladdin Ltd. use {@code etpkcs11.dll}</li>
     * </ul></p>
     * @return 
     *
     * @throws com.pdftools.NotFoundException The library cannot be found.
     * @throws com.pdftools.ExistsException The module has been loaded already by this application.
     * @throws IllegalArgumentException The given {@link com.pdftools.crypto.providers.pkcs11.Module#load library} is not a PKCS#11 driver module.
     * @throws IllegalArgumentException if {@code library} is {@code null}
     */
    public static com.pdftools.crypto.providers.pkcs11.Module load(String library) 
        throws 
            com.pdftools.NotFoundException,

            com.pdftools.ExistsException
    {
        if (library == null)
            throw new IllegalArgumentException("Argument 'library' must not be null.", new NullPointerException("'library'"));

        long retHandle = loadNative(library);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
                case 22: throw new com.pdftools.ExistsException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.crypto.providers.pkcs11.Module.createDynamicObject(retHandle);
    }



     /**
     * <h1>Enable full parallelization (Getter)</h1>
     * <p>
     * The PKCS#11 standard specifies that "an application can specify that it will be accessing the library concurrently from multiple
     * threads, and the library must [...] ensure proper thread-safe behavior."
     * However, some PKCS#11 modules (middleware) implementations are not thread-safe.
     * For this reason, the SDK synchronizes all access to the module.
     * If the middleware is thread-safe, full parallel usage of the cryptographic device can be enabled by setting this property to {@code true}
     * and thereby improving the performance.</p>
     * <p>
     * Default: {@code false}</p>
     */
    public boolean getEnableFullParallelization()
    {
        boolean retVal = getEnableFullParallelizationNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Enable full parallelization (Setter)</h1>
     * <p>
     * The PKCS#11 standard specifies that "an application can specify that it will be accessing the library concurrently from multiple
     * threads, and the library must [...] ensure proper thread-safe behavior."
     * However, some PKCS#11 modules (middleware) implementations are not thread-safe.
     * For this reason, the SDK synchronizes all access to the module.
     * If the middleware is thread-safe, full parallel usage of the cryptographic device can be enabled by setting this property to {@code true}
     * and thereby improving the performance.</p>
     * <p>
     * Default: {@code false}</p>
     */
    public void setEnableFullParallelization(boolean value)
    {
        boolean retVal = setEnableFullParallelizationNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The list of devices managed by this module (Getter)</h1>
     * Most often there is only a single device, so the method {@link DeviceList#getSingle } can be used.
     */
    public com.pdftools.crypto.providers.pkcs11.DeviceList getDevices()
    {
        long retHandle = getDevicesNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.providers.pkcs11.DeviceList.createDynamicObject(retHandle);

    }




    private static native long loadNative(String library);

    private native boolean getEnableFullParallelizationNative(long handle);
    private native boolean setEnableFullParallelizationNative(long handle, boolean value);
    private native long getDevicesNative(long handle);

    /**
     * <h1>Close all open sessions and unload the module</h1>
     * <p>
     * After unloading, the module cannot be loaded again in the same process.
     * Proper unloading is crucial.
     * Otherwise, the application might crash and/or the HSM, USB token, or smart card might not be unlocked.</p>
     * <p>
     * When using the C interface, this method must not be called from the context of the destructor of a global or static object,
     * nor an {@code atexit()} handler, nor the {@code DllMain()} entry point.
     * </p>
     * @throws com.pdftools.PdfToolsException only explicitly stated in a superclass
     */
    public void close() 
        throws com.pdftools.PdfToolsException, 
               java.io.IOException
    {
        try
        {
            if (!close(getHandle()))
                throwLastError();
        }
        finally
        {
            setHandle(0);
        }
    }

    private native boolean close(long hObject);
}

