/****************************************************************************
 *
 * File:            SignatureConfiguration.java
 *
 * Description:     PDFTOOLS SignatureConfiguration Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers.swisscomsigsrv;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The signature configuration</h1>
 */
public class SignatureConfiguration extends com.pdftools.sign.SignatureConfiguration 
{
    protected SignatureConfiguration(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static SignatureConfiguration createDynamicObject(long handle)
    {
        return new SignatureConfiguration(handle);
    }




     /**
     * <h1>The message digest algorithm (Getter)</h1>
     * <p>
     * The algorithm used to hash the document and from which the cryptographic signature is created.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.HashAlgorithm#SHA256 pdftools.crypto.HashAlgorithm.SHA256}</p>
     */
    public com.pdftools.crypto.HashAlgorithm getHashAlgorithm()
    {
        int retVal = getHashAlgorithmNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.HashAlgorithm.fromValue(retVal);
    }

     /**
     * <h1>The message digest algorithm (Setter)</h1>
     * <p>
     * The algorithm used to hash the document and from which the cryptographic signature is created.</p>
     * <p>
     * Default: {@link com.pdftools.crypto.HashAlgorithm#SHA256 pdftools.crypto.HashAlgorithm.SHA256}</p>
     *
     * @throws IllegalArgumentException If the value is invalid or not supported.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setHashAlgorithm(com.pdftools.crypto.HashAlgorithm value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setHashAlgorithmNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The format (encoding) of the cryptographic signature (Getter)</h1>
     * Default: {@link com.pdftools.crypto.SignatureFormat#ADBE_PKCS7_DETACHED pdftools.crypto.SignatureFormat.ADBE_PKCS7_DETACHED}
     */
    public com.pdftools.crypto.SignatureFormat getSignatureFormat()
    {
        int retVal = getSignatureFormatNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.crypto.SignatureFormat.fromValue(retVal);
    }

     /**
     * <h1>The format (encoding) of the cryptographic signature (Setter)</h1>
     * Default: {@link com.pdftools.crypto.SignatureFormat#ADBE_PKCS7_DETACHED pdftools.crypto.SignatureFormat.ADBE_PKCS7_DETACHED}
     *
     * @throws IllegalArgumentException If the value is invalid or not supported.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setSignatureFormat(com.pdftools.crypto.SignatureFormat value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setSignatureFormatNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to add a trusted time-stamp to the signature (Getter)</h1>
     * Default: {@code false}
     */
    public boolean getAddTimestamp()
    {
        boolean retVal = getAddTimestampNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to add a trusted time-stamp to the signature (Setter)</h1>
     * Default: {@code false}
     */
    public void setAddTimestamp(boolean value)
    {
        boolean retVal = setAddTimestampNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to embed validation information into the signature (LTV) (Getter)</h1>
     * <p>
     * <ul>
     * <li>
     * {@code true}: Create an LTV signature by embedding validation information
     * (see {@link com.pdftools.crypto.ValidationInformation#EMBED_IN_SIGNATURE pdftools.crypto.ValidationInformation.EMBED_IN_SIGNATURE}).
     * This value is only supported, if the {@link SignatureConfiguration#getSignatureFormat } is {@link com.pdftools.crypto.SignatureFormat#ADBE_PKCS7_DETACHED pdftools.crypto.SignatureFormat.ADBE_PKCS7_DETACHED}.
     * LTV signatures for other formats can be created by adding validation information to the signed document (see {@link com.pdftools.sign.Signer#process pdftools.sign.Signer.process} and {@link com.pdftools.sign.OutputOptions#getAddValidationInformation pdftools.sign.OutputOptions.getAddValidationInformation}).</li>
     * <li>
     * {@code false}: Create a basic signature without validation information (see {@link com.pdftools.crypto.ValidationInformation#NONE pdftools.crypto.ValidationInformation.NONE}).</li>
     * </ul></p>
     * <p>
     * Default: {@code true}</p>
     */
    public boolean getEmbedValidationInformation()
    {
        boolean retVal = getEmbedValidationInformationNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to embed validation information into the signature (LTV) (Setter)</h1>
     * <p>
     * <ul>
     * <li>
     * {@code true}: Create an LTV signature by embedding validation information
     * (see {@link com.pdftools.crypto.ValidationInformation#EMBED_IN_SIGNATURE pdftools.crypto.ValidationInformation.EMBED_IN_SIGNATURE}).
     * This value is only supported, if the {@link SignatureConfiguration#getSignatureFormat } is {@link com.pdftools.crypto.SignatureFormat#ADBE_PKCS7_DETACHED pdftools.crypto.SignatureFormat.ADBE_PKCS7_DETACHED}.
     * LTV signatures for other formats can be created by adding validation information to the signed document (see {@link com.pdftools.sign.Signer#process pdftools.sign.Signer.process} and {@link com.pdftools.sign.OutputOptions#getAddValidationInformation pdftools.sign.OutputOptions.getAddValidationInformation}).</li>
     * <li>
     * {@code false}: Create a basic signature without validation information (see {@link com.pdftools.crypto.ValidationInformation#NONE pdftools.crypto.ValidationInformation.NONE}).</li>
     * </ul></p>
     * <p>
     * Default: {@code true}</p>
     */
    public void setEmbedValidationInformation(boolean value)
    {
        boolean retVal = setEmbedValidationInformationNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native int getHashAlgorithmNative(long handle);
    private native boolean setHashAlgorithmNative(long handle, int value);
    private native int getSignatureFormatNative(long handle);
    private native boolean setSignatureFormatNative(long handle, int value);
    private native boolean getAddTimestampNative(long handle);
    private native boolean setAddTimestampNative(long handle, boolean value);
    private native boolean getEmbedValidationInformationNative(long handle);
    private native boolean setEmbedValidationInformationNative(long handle, boolean value);

}

