/****************************************************************************
 *
 * File:            StepUp.java
 *
 * Description:     PDFTOOLS StepUp Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.crypto.providers.swisscomsigsrv;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.*;
/**
 * <h1>The options for step-up authorization using Mobile ID</h1>
 */
public class StepUp extends NativeObject 
{
    protected StepUp(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static StepUp createDynamicObject(long handle)
    {
        return new StepUp(handle);
    }

    /***
     * Listener interface for the {@link ConsentRequired} event.
     *
     */
    public interface ConsentRequiredListener extends EventListener
    {
        void consentRequired(ConsentRequired event);
    }

    /**
     * 
     * Event containing the URL for step-up authentication using password and SMS challenge (OTP).
     * Password and SMS challenge are used as a fallback mechanism for the Mobile ID authentication.
     * For example, if the Mobile ID of the user is not activated.
     * The user must be redirected to this URL for consent of will.
     */
    public class ConsentRequired extends EventObject
    {
        private static final long serialVersionUID = 1563L;

        private java.net.URI url;

        private ConsentRequired(Object source, String url) {
            super(source);
            this.url = java.net.URI.create(url);

        }

        /**
         * The consent URL where the user must be redirected to acknowledge the declaration of will.
         */
        public java.net.URI getUrl() {
            return url;
        }


    }

    private Hashtable<ConsentRequiredListener, ConsentRequiredNativeClass> consentRequiredDic = new Hashtable<ConsentRequiredListener, ConsentRequiredNativeClass>();

    private class ConsentRequiredNativeClass
    {
        private ConsentRequiredListener listener;
        private long context;
        public ConsentRequiredNativeClass(ConsentRequiredListener listener)
        {
            this.listener = listener;
        }

        public void consentRequiredHandler(String url)
        {
            ConsentRequired event = new ConsentRequired(this, url);
            try {
                this.listener.consentRequired(event);
            }
            catch (Exception ex) { }
        }
    }

    /**
     * Add a listener for the {@link ConsentRequired} event.
     * @param listener Listener for the {@link ConsentRequired} event. 
     *                 If a listener is added that is already registered, it is ignored.
     */
    public void addConsentRequiredListener(ConsentRequiredListener listener)
    {
        if(!consentRequiredDic.containsKey(listener))
        {
            ConsentRequiredNativeClass eventNativeClass = new ConsentRequiredNativeClass(listener);
            long context = addConsentRequiredHandlerNative(getHandle(), eventNativeClass);
            if (context == 0)
                throwLastRuntimeException();

            eventNativeClass.context = context;
            consentRequiredDic.put(listener, eventNativeClass);
        }
    }

    /**
     * Remove registered listener for the {@link ConsentRequired} event.
     * @param listener Listener for the {@link ConsentRequired} event that should be removed.
     *                 If the listener is not registered, it is ignored.
     */
    public void removeConsentRequiredListener(ConsentRequiredListener listener)
    {
        if(consentRequiredDic.containsKey(listener))
        {
            if (!removeConsentRequiredHandlerNative(getHandle(), consentRequiredDic.get(listener).context))
            {
                if (getLastErrorCode() != 5)
                    throwLastRuntimeException();
            }
            consentRequiredDic.remove(listener);
        }
    }



    /**
     * 
     * @param msisdn 
     * The mobile phone number
     * @param message 
     * The message to be displayed on the mobile phone
     * @param language 
     * The language of the message
     * @throws IllegalArgumentException if {@code msisdn} is {@code null}
     * @throws IllegalArgumentException if {@code message} is {@code null}
     * @throws IllegalArgumentException if {@code language} is {@code null}
     */
    public StepUp(String msisdn, String message, String language)
    {
        this(newHelper(msisdn, message, language));
    }

    private static long newHelper(String msisdn, String message, String language)
    {
        if (msisdn == null)
            throw new IllegalArgumentException("Argument 'msisdn' must not be null.", new NullPointerException("'msisdn'"));
        if (message == null)
            throw new IllegalArgumentException("Argument 'message' must not be null.", new NullPointerException("'message'"));
        if (language == null)
            throw new IllegalArgumentException("Argument 'language' must not be null.", new NullPointerException("'language'"));

        long handle = newNative(msisdn, message, language);
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The mobile phone number (Getter)</h1>
     * Example: {@code "+41798765432"}
     */
    public String getMSISDN()
    {
        String retVal = getMSISDNNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The mobile phone number (Setter)</h1>
     * Example: {@code "+41798765432"}
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setMSISDN(String value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setMSISDNNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The message to be displayed on the mobile phone (Getter)</h1>
     * Example: {@code "Do you authorize your signature on Contract.pdf?"}
     */
    public String getMessage()
    {
        String retVal = getMessageNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The message to be displayed on the mobile phone (Setter)</h1>
     * Example: {@code "Do you authorize your signature on Contract.pdf?"}
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setMessage(String value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setMessageNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The language of the message (Getter)</h1>
     * Example: {@code "DE"}
     */
    public String getLanguage()
    {
        String retVal = getLanguageNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The language of the message (Setter)</h1>
     * Example: {@code "DE"}
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setLanguage(String value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setLanguageNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native long addConsentRequiredHandlerNative(long handle, ConsentRequiredNativeClass eventClass);
    private native boolean removeConsentRequiredHandlerNative(long handle, long context);

    private static native long newNative(String msisdn, String message, String language);
    private native String getMSISDNNative(long handle);
    private native boolean setMSISDNNative(long handle, String value);
    private native String getMessageNative(long handle);
    private native boolean setMessageNative(long handle, String value);
    private native String getLanguageNative(long handle);
    private native boolean setLanguageNative(long handle, String value);

}

