/****************************************************************************
 *
 * File:            DocumentCopyOptions.java
 *
 * Description:     PDFTOOLS DocumentCopyOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.documentassembly;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * 
 * The document-level copy options applied when copying a document.
 */
public class DocumentCopyOptions extends NativeObject 
{
    protected DocumentCopyOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static DocumentCopyOptions createDynamicObject(long handle)
    {
        return new DocumentCopyOptions(handle);
    }


    /**
     * 
     */
    public DocumentCopyOptions()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <p>
     * Copy document information dictionary and XMP metadata.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public boolean getCopyMetadata()
    {
        boolean retVal = getCopyMetadataNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <p>
     * Copy document information dictionary and XMP metadata.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public void setCopyMetadata(boolean value)
    {
        boolean retVal = setCopyMetadataNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <p>
     * Copy the PDF/A output intent.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public boolean getCopyOutputIntent()
    {
        boolean retVal = getCopyOutputIntentNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <p>
     * Copy the PDF/A output intent.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public void setCopyOutputIntent(boolean value)
    {
        boolean retVal = setCopyOutputIntentNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <p>
     * Copy viewer properties, which include: Page Layout, Page Mode, Open Actions, Piece Info, and Collection properties.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public boolean getCopyViewerSettings()
    {
        boolean retVal = getCopyViewerSettingsNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <p>
     * Copy viewer properties, which include: Page Layout, Page Mode, Open Actions, Piece Info, and Collection properties.</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public void setCopyViewerSettings(boolean value)
    {
        boolean retVal = setCopyViewerSettingsNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <p>
     * If set to {@code true}: All embedded files are copied. If set to {@code false}: Only embedded files associated with pages
     * within the given page range are copied. (PDF/A-3 only, {@link PageCopyOptions#getCopyAssociatedFiles } must be set.)</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public boolean getCopyEmbeddedFiles()
    {
        boolean retVal = getCopyEmbeddedFilesNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <p>
     * If set to {@code true}: All embedded files are copied. If set to {@code false}: Only embedded files associated with pages
     * within the given page range are copied. (PDF/A-3 only, {@link PageCopyOptions#getCopyAssociatedFiles } must be set.)</p>
     * <p>
     * Default: {@code false}.
     * </p>
     */
    public void setCopyEmbeddedFiles(boolean value)
    {
        boolean retVal = setCopyEmbeddedFilesNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native boolean getCopyMetadataNative(long handle);
    private native boolean setCopyMetadataNative(long handle, boolean value);
    private native boolean getCopyOutputIntentNative(long handle);
    private native boolean setCopyOutputIntentNative(long handle, boolean value);
    private native boolean getCopyViewerSettingsNative(long handle);
    private native boolean setCopyViewerSettingsNative(long handle, boolean value);
    private native boolean getCopyEmbeddedFilesNative(long handle);
    private native boolean setCopyEmbeddedFilesNative(long handle, boolean value);

}

